<?php

namespace App\Services;

use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\Setting;
use App\Models\User;

class UserCoinService
{
    /**
     * Ensure user has coin balance records for all active coins
     * This method creates UserCoinBalance records for any missing active coins
     * and populates addresses from wallet settings. Also updates existing records
     * that have null addresses.
     */
    public function ensureUserCoinBalances(User $user): void
    {
        // Get all active coins
        $activeCoins = Coin::where('is_active', true)->get();
        
        // Get existing coin balances for this user
        $existingBalances = UserCoinBalance::where('user_id', $user->id)
            ->get()
            ->keyBy('coin_id');
        
        foreach ($activeCoins as $coin) {
            // Generate the wallet address setting key based on coin and network
            $settingKey = $this->generateWalletSettingKey($coin);
            
            // Get address from wallet settings
            $walletAddress = Setting::where('key', $settingKey)->value('value');
            
            if (isset($existingBalances[$coin->id])) {
                // Update existing record if address is null and we have a wallet address
                $existingBalance = $existingBalances[$coin->id];
                if (is_null($existingBalance->address) && !is_null($walletAddress)) {
                    $existingBalance->update(['address' => $walletAddress]);
                }
            } else {
                // Create new user coin balance record
                UserCoinBalance::create([
                    'user_id' => $user->id,
                    'coin_id' => $coin->id,
                    'balance' => 0,
                    'address' => $walletAddress, // Use address from wallet settings
                    'is_enabled' => true
                ]);
            }
        }
    }

    /**
     * Create coin balances for a new user (used during registration)
     * This method creates UserCoinBalance records for ALL active coins
     */
    public function createUserCoinBalances(User $user): void
    {
        // Get all active coins
        $activeCoins = Coin::where('is_active', true)->get();
        
        foreach ($activeCoins as $coin) {
            // Generate the wallet address setting key based on coin and network
            $settingKey = $this->generateWalletSettingKey($coin);
            
            // Get address from wallet settings
            $walletAddress = Setting::where('key', $settingKey)->value('value');
            
            // Create user coin balance record
            UserCoinBalance::create([
                'user_id' => $user->id,
                'coin_id' => $coin->id,
                'balance' => 0,
                'address' => $walletAddress, // Use address from wallet settings
                'is_enabled' => true
            ]);
        }
    }

    /**
     * Generate the wallet setting key for a coin
     * Examples: btc_address, usdt_trc20_address, eth_address
     */
    public function generateWalletSettingKey(Coin $coin): string
    {
        $settingKey = strtolower($coin->symbol);
        
        if ($coin->network) {
            $settingKey .= '_' . strtolower($coin->network);
        }
        
        $settingKey .= '_address';
        
        return $settingKey;
    }

    /**
     * Get the user's coin balance for a specific coin
     */
    public function getUserCoinBalance(User $user, Coin $coin): ?UserCoinBalance
    {
        return UserCoinBalance::where('user_id', $user->id)
            ->where('coin_id', $coin->id)
            ->first();
    }

    /**
     * Create a single coin balance for a user
     */
    public function createUserCoinBalance(User $user, Coin $coin): UserCoinBalance
    {
        // Generate the wallet address setting key based on coin and network
        $settingKey = $this->generateWalletSettingKey($coin);
        
        // Get address from wallet settings
        $walletAddress = Setting::where('key', $settingKey)->value('value');
        
        // Create user coin balance record
        return UserCoinBalance::create([
            'user_id' => $user->id,
            'coin_id' => $coin->id,
            'balance' => 0,
            'address' => $walletAddress, // Use address from wallet settings
            'is_enabled' => true
        ]);
    }

    /**
     * Get all active coin balances for a user
     */
    public function getUserActiveCoins(User $user)
    {
        return UserCoinBalance::where('user_id', $user->id)
            ->where('is_enabled', true)
            ->with('coin')
            ->get();
    }

    /**
     * Update all users' coin balances that have null addresses
     * with current wallet settings addresses
     */
    public function updateNullAddressesFromWalletSettings(): int
    {
        $updatedCount = 0;
        
        // Get all active coins
        $activeCoins = Coin::where('is_active', true)->get();
        
        foreach ($activeCoins as $coin) {
            // Generate the wallet address setting key
            $settingKey = $this->generateWalletSettingKey($coin);
            
            // Get address from wallet settings
            $walletAddress = Setting::where('key', $settingKey)->value('value');
            
            // Skip if no wallet address is configured
            if (is_null($walletAddress)) {
                continue;
            }
            
            // Update all UserCoinBalance records for this coin that have null addresses
            $updated = UserCoinBalance::where('coin_id', $coin->id)
                ->whereNull('address')
                ->update(['address' => $walletAddress]);
                
            $updatedCount += $updated;
        }
        
        return $updatedCount;
    }
} 