<?php

namespace App\Services;

use App\Models\User;
use App\Models\Transaction;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UserActivityService
{
    /**
     * Log user activity and send email notification to admin
     *
     * @param string $activity The activity type
     * @param User $user The user performing the activity
     * @param array $data Additional data about the activity
     * @param string|null $description A custom description of the activity
     * @return bool Whether the operation was successful
     */
    public static function log(string $activity, User $user, array $data = [], ?string $description = null): bool
    {
        $adminEmail = setting('company_email');
        
        if (empty($adminEmail)) {
            Log::error('Admin email not configured. Set company_email in settings.');
            return false;
        }

        // Prepare activity data
        $activityData = [
            'activity' => $activity,
            'user_id' => $user->id,
            'user_email' => $user->email,
            'user_name' => $user->name,
            'timestamp' => now()->format('Y-m-d H:i:s'),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'data' => $data
        ];
        
        // Set description based on activity type if not provided
        if (!$description) {
            $description = self::getActivityDescription($activity, $user, $data);
        }
        
        $activityData['description'] = $description;
        
        try {
            // Send email to admin
            Mail::send('emails.user_activity', ['activity' => $activityData], function ($message) use ($adminEmail, $activity, $user) {
                $message->to($adminEmail)
                    ->subject("User Activity: {$activity} - {$user->email}");
            });
            
            // Log the activity for debugging purposes
            Log::info('User activity', $activityData);
            
            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send user activity email', [
                'error' => $e->getMessage(),
                'activity' => $activityData
            ]);
            return false;
        }
    }
    
    /**
     * Log activity within a database transaction
     * Will roll back the transaction if email notification fails
     *
     * @param string $activity Activity type
     * @param User $user User performing the activity
     * @param callable $callback The database operation to perform
     * @param array $data Additional data about the activity
     * @param string|null $description Custom description
     * @return mixed The result of the callback or null if failed
     * @throws \Exception If the operation fails and the transaction is rolled back
     */
    public static function logInTransaction(string $activity, User $user, callable $callback, array $data = [], ?string $description = null)
    {
        return DB::transaction(function () use ($activity, $user, $callback, $data, $description) {
            // Execute the callback first to get its result
            $result = $callback();
            
            // If there's transaction data from the result, add it to the activity data
            if (isset($result) && $result instanceof Transaction) {
                $data['transaction_id'] = $result->id;
                $data['transaction_type'] = $result->type;
                $data['transaction_amount'] = $result->amount_in;
                $data['transaction_status'] = $result->status;
            }
            
            // Log the activity and send email
            $logSuccess = self::log($activity, $user, $data, $description);
            
            // Throw exception to trigger rollback if log fails
            if (!$logSuccess) {
                throw new \Exception("Failed to log user activity: {$activity}");
            }
            
            return $result;
        });
    }
    
    /**
     * Generate a description for the activity
     *
     * @param string $activity Activity type
     * @param User $user User performing the activity
     * @param array $data Activity data
     * @return string Description
     */
    private static function getActivityDescription(string $activity, User $user, array $data): string
    {
        switch ($activity) {
            case 'login':
                return "User {$user->email} logged in";
                
            case 'logout':
                return "User {$user->email} logged out";
                
            case 'register':
                return "New user {$user->email} registered";
                
            case 'deposit':
                $amount = $data['amount'] ?? 'unknown amount';
                $currency = $data['currency'] ?? 'unknown currency';
                return "User {$user->email} deposited {$amount} {$currency}";
                
            case 'withdraw':
                $amount = $data['amount'] ?? 'unknown amount';
                $currency = $data['currency'] ?? 'unknown currency';
                return "User {$user->email} withdrew {$amount} {$currency}";
                
            case 'swap':
                $fromAmount = $data['from_amount'] ?? 'unknown amount';
                $fromCurrency = $data['from_currency'] ?? 'unknown currency';
                $toAmount = $data['to_amount'] ?? 'unknown amount';
                $toCurrency = $data['to_currency'] ?? 'unknown currency';
                return "User {$user->email} swapped {$fromAmount} {$fromCurrency} to {$toAmount} {$toCurrency}";
                
            case 'profile_update':
                return "User {$user->email} updated their profile";
                
            case 'password_update':
                return "User {$user->email} updated their password";
                
            case 'kyc_submit':
                return "User {$user->email} submitted KYC documents";
                
            case 'bot_subscribe':
                $botName = $data['bot_name'] ?? 'unknown bot';
                $amount = $data['amount'] ?? 'unknown amount';
                return "User {$user->email} subscribed to {$botName} bot with {$amount}";
                
            case 'stake':
                $amount = $data['amount'] ?? 'unknown amount';
                $currency = $data['currency'] ?? 'unknown currency';
                $plan = $data['plan_name'] ?? 'unknown plan';
                $roi = $data['roi'] ?? 'unknown ROI';
                $interestType = $data['interest_type'] ?? 'unknown type';
                $lock = $data['lock_duration'] ?? 'unknown duration';
                return "User {$user->email} staked {$amount} {$currency} in {$plan} (ROI: {$roi}%, {$interestType}, Lock: {$lock} days)";
                
            default:
                return "User {$user->email} performed {$activity}";
        }
    }
} 