<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Coin;
use Exception;

class ExchangeSupportChecker
{
    private $exchanges = [
        'huobi' => [
            'url' => 'https://api.huobi.pro/v1/common/symbols',
            'method' => 'GET'
        ],
        'gateio' => [
            'url' => 'https://api.gateio.ws/api/v4/spot/currency_pairs',
            'method' => 'GET'
        ],
        'coingecko' => [
            'url' => 'https://api.coingecko.com/api/v3/coins/list',
            'method' => 'GET'
        ]
    ];

    /**
     * Check support for a specific coin across all exchanges
     */
    public function checkCoinSupport(Coin $coin): array
    {
        $results = [];
        
        foreach ($this->exchanges as $exchange => $config) {
            try {
                $results[$exchange] = $this->checkExchangeSupport($coin, $exchange, $config);
            } catch (Exception $e) {
                Log::error("Error checking {$exchange} support for {$coin->symbol}", [
                    'error' => $e->getMessage(),
                    'coin_id' => $coin->id
                ]);
                $results[$exchange] = [
                    'supported' => false,
                    'error' => $e->getMessage(),
                    'last_checked' => now()
                ];
            }
        }

        return $results;
    }

    /**
     * Check support for a coin on a specific exchange
     */
    private function checkExchangeSupport(Coin $coin, string $exchange, array $config): array
    {
        $response = Http::timeout(10)
            ->retry(2, 100)
            ->get($config['url']);

        if (!$response->successful()) {
            throw new Exception("API request failed with status: " . $response->status());
        }

        $data = $response->json();
        $supported = $this->parseExchangeResponse($coin, $exchange, $data);

        return [
            'supported' => $supported,
            'last_checked' => now(),
            'response_size' => count($data ?? [])
        ];
    }

    /**
     * Parse exchange API response to check coin support
     */
    private function parseExchangeResponse(Coin $coin, string $exchange, array $data): bool
    {
        return match($exchange) {
            'huobi' => $this->parseHuobiResponse($coin, $data), 
            'gateio' => $this->parseGateioResponse($coin, $data),
            'coingecko' => $this->parseCoingeckoResponse($coin, $data),
            default => false
        };
    }

    /**
     * Parse Huobi response
     */
    private function parseHuobiResponse(Coin $coin, array $data): bool
    {
        if (!isset($data['data']) || !is_array($data['data'])) {
            return false;
        }

        $symbols = collect($data['data'])
            ->where('quote-currency', 'usdt')
            ->pluck('base-currency')
            ->map('strtoupper');

        return $symbols->contains($coin->symbol);
    }

    /**
     * Parse Gate.io response
     */
    private function parseGateioResponse(Coin $coin, array $data): bool
    {
        if (!is_array($data)) {
            return false;
        }

        $symbols = collect($data)
            ->filter(function ($pair) {
                return str_ends_with($pair['id'], '_USDT');
            })
            ->map(function ($pair) {
                return strtoupper(explode('_', $pair['id'])[0]);
            });

        return $symbols->contains($coin->symbol);
    }

    /**
     * Parse CoinGecko response
     */
    private function parseCoingeckoResponse(Coin $coin, array $data): bool
    {
        if (!is_array($data)) {
            return false;
        }

        $priceSymbols = collect($data)->pluck('id');
        return $priceSymbols->contains($coin->price_symbol);
    }

    /**
     * Update coin support status in database
     */
    public function updateCoinSupportStatus(Coin $coin): array
    {
        $results = $this->checkCoinSupport($coin);
        
        // Count supported exchanges
        $supportedCount = collect($results)->where('supported', true)->count();
        $totalCount = count($results);
        
        // Determine overall status
        $status = match(true) {
            $supportedCount >= 3 => 'excellent',
            $supportedCount >= 2 => 'good',
            $supportedCount >= 1 => 'limited',
            default => 'unsupported'
        };

        // Update coin record
        $coin->update([
            'supported_exchanges' => $results,
            'last_price_check' => now(),
            'price_check_status' => $supportedCount > 0 ? 'supported' : 'unsupported'
        ]);

        return [
            'coin' => $coin->fresh(),
            'support_results' => $results,
            'status' => $status,
            'supported_count' => $supportedCount,
            'total_count' => $totalCount,
            'recommendations' => $this->generateRecommendations($coin, $results, $status)
        ];
    }

    /**
     * Check all coins support status
     */
    public function checkAllCoins(): array
    {
        $coins = Coin::active()->get();
        $results = [];

        foreach ($coins as $coin) {
            try {
                $results[] = $this->updateCoinSupportStatus($coin);
            } catch (Exception $e) {
                Log::error("Failed to check support for coin {$coin->symbol}", [
                    'error' => $e->getMessage(),
                    'coin_id' => $coin->id
                ]);
                
                $results[] = [
                    'coin' => $coin,
                    'error' => $e->getMessage(),
                    'status' => 'error'
                ];
            }
        }

        return $results;
    }

    /**
     * Generate recommendations based on support results
     */
    private function generateRecommendations(Coin $coin, array $results, string $status): array
    {
        $recommendations = [];

        switch($status) {
            case 'unsupported':
                $recommendations[] = [
                    'type' => 'critical',
                    'message' => 'This coin is not supported by any exchange. Consider enabling manual pricing or removing it.',
                    'action' => 'enable_manual_pricing'
                ];
                break;
                
            case 'limited':
                $recommendations[] = [
                    'type' => 'warning', 
                    'message' => 'Limited exchange support detected. Monitor pricing accuracy closely.',
                    'action' => 'monitor_pricing'
                ];
                break;
                
            case 'good':
                $recommendations[] = [
                    'type' => 'info',
                    'message' => 'Good exchange support. Consider this coin reliable for pricing.',
                    'action' => 'none'
                ];
                break;
                
            case 'excellent':
                $recommendations[] = [
                    'type' => 'success',
                    'message' => 'Excellent exchange support across multiple platforms.',
                    'action' => 'none'
                ];
                break;
        }

        // Check for CoinGecko fallback availability
        if (isset($results['coingecko']) && $results['coingecko']['supported']) {
            $recommendations[] = [
                'type' => 'info',
                'message' => 'CoinGecko pricing available as fallback option.',
                'action' => 'enable_coingecko_fallback'
            ];
        }

        return $recommendations;
    }
} 