<?php

namespace App\Observers;

use App\Models\Coin;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class CoinObserver
{
    /**
     * Handle the Coin "created" event.
     */
    public function created(Coin $coin): void
    {
        $this->clearCaches();
        Log::info("Coin created: {$coin->symbol}", ['coin_id' => $coin->id]);
    }

    /**
     * Handle the Coin "updated" event.
     */
    public function updated(Coin $coin): void
    {
        $this->clearCaches();
        
        // Handle image file cleanup if image was changed
        if ($coin->isDirty('image') && $coin->getOriginal('image')) {
            $this->cleanupOldImage($coin->getOriginal('image'));
        }
        
        Log::info("Coin updated: {$coin->symbol}", [
            'coin_id' => $coin->id,
            'dirty_fields' => array_keys($coin->getDirty())
        ]);
    }

    /**
     * Handle the Coin "deleted" event.
     */
    public function deleted(Coin $coin): void
    {
        $this->clearCaches();
        
        // Cleanup image file when coin is deleted
        if ($coin->image) {
            $this->cleanupOldImage($coin->image);
        }
        
        Log::info("Coin deleted: {$coin->symbol}", ['coin_id' => $coin->id]);
    }

    /**
     * Handle the Coin "restored" event.
     */
    public function restored(Coin $coin): void
    {
        $this->clearCaches();
        Log::info("Coin restored: {$coin->symbol}", ['coin_id' => $coin->id]);
    }

    /**
     * Handle the Coin "force deleted" event.
     */
    public function forceDeleted(Coin $coin): void
    {
        $this->clearCaches();
        
        // Cleanup image file when coin is force deleted
        if ($coin->image) {
            $this->cleanupOldImage($coin->image);
        }
        
        Log::info("Coin force deleted: {$coin->symbol}", ['coin_id' => $coin->id]);
    }

    /**
     * Clear all related caches
     */
    private function clearCaches(): void
    {
        Cache::forget('active_coins');
        Cache::forget('symbol_mapping');
        Cache::forget('crypto_prices');
        
        // Clear any other coin-related caches
        $cacheKeys = ['coins_for_display', 'coin_icons', 'network_icons'];
        foreach ($cacheKeys as $key) {
            Cache::forget($key);
        }
    }

    /**
     * Clean up old image files
     */
    private function cleanupOldImage(string $imagePath): void
    {
        // Only cleanup if it's a local file (not a URL)
        if (!filter_var($imagePath, FILTER_VALIDATE_URL)) {
            try {
                if (Storage::disk('public')->exists($imagePath)) {
                    Storage::disk('public')->delete($imagePath);
                    Log::info("Cleaned up image file: {$imagePath}");
                }
            } catch (\Exception $e) {
                Log::error("Failed to cleanup image file: {$imagePath}", [
                    'error' => $e->getMessage()
                ]);
            }
        }
    }
}
