<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserCoinBalance extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'coin_id',
        'balance',
        'address',
        'is_enabled',
        'custom_fee',
        'custom_swap_fee'
    ];

    protected $casts = [
        'balance' => 'decimal:8',
        'custom_fee' => 'decimal:8',
        'custom_swap_fee' => 'decimal:8',
        'is_enabled' => 'boolean'
    ];

    /**
     * Relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function coin()
    {
        return $this->belongsTo(Coin::class);
    }

    /**
     * Get the effective fee (custom or coin default)
     */
    public function getEffectiveFee(): float
    {
        return $this->custom_fee ?? $this->coin->default_fee ?? 0;
    }

    /**
     * Get the effective swap fee (custom or coin default)
     */
    public function getEffectiveSwapFee(): float
    {
        return $this->custom_swap_fee ?? $this->coin->default_swap_fee ?? 0;
    }

    /**
     * Check if this balance has a positive amount
     */
    public function hasBalance(): bool
    {
        return $this->balance > 0;
    }

    /**
     * Check if this coin is enabled and has an address
     */
    public function isActive(): bool
    {
        return $this->is_enabled && !empty($this->address);
    }

    /**
     * Get the USD value of this balance
     */
    public function getUsdValue(): float
    {
        if (!$this->hasBalance()) {
            return 0;
        }

        $price = $this->coin->getEffectivePrice();
        return $this->balance * $price;
    }

    /**
     * Scopes
     */
    public function scopeEnabled($query)
    {
        return $query->where('is_enabled', true);
    }

    public function scopeWithBalance($query)
    {
        return $query->where('balance', '>', 0);
    }

    public function scopeActive($query)
    {
        return $query->where('is_enabled', true)
                    ->whereNotNull('address')
                    ->where('address', '!=', '');
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeForCoin($query, $coinId)
    {
        return $query->where('coin_id', $coinId);
    }

    /**
     * Boot method for model events
     */
    protected static function boot()
    {
        parent::boot();

        // Ensure balance is never negative
        static::saving(function ($balance) {
            if ($balance->balance < 0) {
                $balance->balance = 0;
            }
        });
    }
}
