<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Transaction extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'from_coin_id',
        'to_coin_id',
        'type',
        'which_crypto', // Legacy field - to be deprecated
        'transaction_hash',
        'from_address',
        'to_address',
        'amount_in',
        'amount_out',
        'network_fee',
        'rate',
        'status',
        'metadata',
        'related_transaction_id',
        'processed_at',
        // Legacy fields kept for backward compatibility with existing data
        'cryptoasset_id',
        'from_crypto',
        'to_crypto',
    ];

    protected $casts = [
        'amount_in' => 'decimal:8',
        'amount_out' => 'decimal:8',
        'network_fee' => 'decimal:8',
        'rate' => 'decimal:8',
        'metadata' => 'json',
        'processed_at' => 'datetime',
    ];

    // Transaction types
    const TYPE_DEPOSIT = 'deposit';
    const TYPE_WITHDRAWAL = 'withdrawal';
    const TYPE_SWAP = 'swap';
    const TYPE_FUNDING = 'funding';
    const TYPE_REFUND = 'refund';

    // Transaction statuses
    const STATUS_PENDING = 'pending';
    const STATUS_PROCESSING = 'processing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_REVERSED = 'reversed';

    /**
     * Get all valid transaction types
     */
    public static function getValidTypes(): array
    {
        return [
            self::TYPE_DEPOSIT,
            self::TYPE_WITHDRAWAL,
            self::TYPE_SWAP,
            self::TYPE_FUNDING,
            self::TYPE_REFUND,
        ];
    }

    /**
     * Get all valid transaction statuses
     */
    public static function getValidStatuses(): array
    {
        return [
            self::STATUS_PENDING,
            self::STATUS_PROCESSING,
            self::STATUS_COMPLETED,
            self::STATUS_FAILED,
            self::STATUS_CANCELLED,
            self::STATUS_REVERSED,
        ];
    }

    /**
     * Relationships
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function fromCoin()
    {
        return $this->belongsTo(Coin::class, 'from_coin_id');
    }

    public function toCoin()
    {
        return $this->belongsTo(Coin::class, 'to_coin_id');
    }

    public function relatedTransaction()
    {
        return $this->belongsTo(self::class, 'related_transaction_id');
    }

    /**
     * Scopes
     */
    public function scopeOfType($query, string $type)
    {
        return $query->where('type', $type);
    }

    public function scopeOfStatus($query, string $status)
    {
        return $query->where('status', $status);
    }

    public function scopeOfCoin($query, int $coinId)
    {
        return $query->where(function ($q) use ($coinId) {
            $q->where('from_coin_id', $coinId)
              ->orWhere('to_coin_id', $coinId);
        });
    }

    public function scopeSwaps($query)
    {
        return $query->where('type', self::TYPE_SWAP);
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', self::STATUS_COMPLETED);
    }

    public function scopePending($query)
    {
        return $query->where('status', self::STATUS_PENDING);
    }

    /**
     * Status check methods
     */
    public function isCompleted(): bool
    {
        return $this->status === self::STATUS_COMPLETED;
    }

    public function isPending(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    public function isFailed(): bool
    {
        return $this->status === self::STATUS_FAILED;
    }

    public function isProcessing(): bool
    {
        return $this->status === self::STATUS_PROCESSING;
    }

    /**
     * Type check methods
     */
    public function isSwap(): bool
    {
        return $this->type === self::TYPE_SWAP;
    }

    public function isDeposit(): bool
    {
        return $this->type === self::TYPE_DEPOSIT;
    }

    public function isWithdrawal(): bool
    {
        return $this->type === self::TYPE_WITHDRAWAL;
    }

    /**
     * Get the primary coin for this transaction
     */
    public function getPrimaryCoin()
    {
        return $this->fromCoin ?? $this->toCoin;
    }

    /**
     * Get formatted transaction description using coin system
     */
    public function getDescription(): string
    {
        if ($this->isSwap() && $this->fromCoin && $this->toCoin) {
            return "Swapped {$this->amount_in} {$this->fromCoin->symbol} to {$this->amount_out} {$this->toCoin->symbol}";
        }

        if ($this->isDeposit() && $this->fromCoin) {
            return "Deposited {$this->amount_in} {$this->fromCoin->symbol}";
        }

        if ($this->isWithdrawal() && $this->fromCoin) {
            // Check for KYC fee in metadata
            if ($this->metadata && isset($this->metadata['purpose']) && $this->metadata['purpose'] === 'kyc_fee') {
                return "KYC verification fee - {$this->amount_in} {$this->fromCoin->symbol}";
            }
            return "Withdrew {$this->amount_in} {$this->fromCoin->symbol}";
        }

        if ($this->type === self::TYPE_FUNDING && $this->fromCoin) {
            // Check for card funding in metadata
            if ($this->metadata && isset($this->metadata['purpose']) && $this->metadata['purpose'] === 'card_funding') {
                return "Card funding - {$this->amount_in} {$this->fromCoin->symbol}";
            }
            return "Funded {$this->amount_in} {$this->fromCoin->symbol}";
        }

        // Fallback for transactions without proper coin relationships
        $coinSymbol = $this->fromCoin?->symbol ?? $this->toCoin?->symbol ?? $this->which_crypto ?? 'Unknown';
        return ucfirst($this->type) . " of {$this->amount_in} {$coinSymbol}";
    }

    /**
     * Get the transaction fee information
     */
    public function getFeeInfo(): ?array
    {
        // Check for swap fees in metadata
        if ($this->metadata && isset($this->metadata['swap_fee_amount']) && $this->metadata['swap_fee_amount'] > 0) {
            return [
                'amount' => $this->metadata['swap_fee_amount'],
                'currency' => $this->fromCoin?->symbol ?? 'Unknown',
                'usd_value' => $this->metadata['swap_fee_usd'] ?? 0,
                'type' => 'swap_fee'
            ];
        }

        // Check for network fees
        if ($this->network_fee && $this->network_fee > 0) {
            return [
                'amount' => $this->network_fee,
                'currency' => $this->fromCoin?->symbol ?? 'Unknown',
                'usd_value' => $this->network_fee * ($this->fromCoin?->getEffectivePrice() ?? 0),
                'type' => 'network_fee'
            ];
        }

        return null;
    }

    /**
     * Get the USD value of this transaction
     */
    public function getUsdValue(): float
    {
        if ($this->metadata && isset($this->metadata['usd_value'])) {
            return (float) $this->metadata['usd_value'];
        }

        // Calculate from coin prices if not stored
        if ($this->fromCoin && $this->amount_in) {
            return $this->amount_in * $this->fromCoin->getEffectivePrice();
        }

        return 0;
    }

    /**
     * Check if this transaction involves a specific coin
     */
    public function involvesCoin(int $coinId): bool
    {
        return $this->from_coin_id === $coinId || $this->to_coin_id === $coinId;
    }
}