<?php

namespace App\Mail;

use App\Models\Transaction;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;

class TransactionStatusEmail extends Mailable
{
    use Queueable, SerializesModels;

    public $transaction;
    protected $status;
    protected $reason;

    /**
     * Create a new message instance.
     */
    public function __construct(Transaction $transaction, string $status, ?string $reason = null)
    {
        $this->transaction = $transaction;
        $this->status = $status;
        $this->reason = $reason;
    }

    /**
     * Build the message.
     */
    public function build()
    {
        return $this->subject($this->getSubject())
            ->markdown('emails.transaction-status')
            ->with([
                'transaction'  => $this->transaction,
                'status'       => $this->status,
                'reason'       => $this->reason,
                'type'         => $this->transaction->type,
                'amount'       => $this->getAmount(),
                'crypto'       => $this->getCrypto(),
                'fee'          => $this->getFee(),
                'hash'         => $this->transaction->transaction_hash,
                'network'      => $this->transaction->network ?? null,
                'metadata'     => $this->transaction->metadata ?? [],
                'companyName'  => setting('company_name', config('app.name')),
                'supportEmail' => setting('company_email', 'support@example.com'),
                'created_at'   => $this->transaction->created_at,
            ]);
    }

    /**
     * Get email subject based on transaction type and status.
     */
    private function getSubject(): string
    {
        $type   = ucfirst($this->transaction->type);
        $crypto = strtoupper($this->getCrypto());
        $amount = $this->formatAmountForSubject();

        return match ($this->status) {
            'completed' => " {$type} of {$amount} {$crypto} Completed",
            'pending'   => " {$type} of {$amount} {$crypto} Pending",
            'failed'    => " {$type} of {$amount} {$crypto} Failed",
            default     => " {$type} Transaction Update: {$amount} {$crypto}",
        };
    }

    /**
     * Get the transaction amount based on type.
     *
     * - withdrawal → amount_out
     * - deposit → amount_in
     * - swap → amount_in (base)
     */
    private function getAmount(): float
    {
        return match ($this->transaction->type) {
            'withdrawal' => $this->transaction->amount_out 
                            ?? 0.0, // Send
            'deposit'    => $this->transaction->amount_in 
                            ?? 0.0, // Receive
            'swap'       => $this->transaction->amount_in 
                            ?? 0.0, // Base
            default      => $this->transaction->amount_in 
                            ?? $this->transaction->amount_out 
                            ?? 0.0,
        };
    }

    /**
     * Format amount for subject line (adds 2 decimals for fiat, 8 for crypto).
     */
    private function formatAmountForSubject(): string
    {
        $amount = $this->getAmount();

        // Usually crypto amounts should show 8 decimals for precision
        return number_format($amount, 8);
    }

    /**
     * Get cryptocurrency symbol (if any).
     */
    private function getCrypto(): string
    {
        return strtoupper(
            $this->transaction->which_crypto
            ?? ($this->transaction->fromCoin->display_symbol ?? null)
            ?? ($this->transaction->toCoin->display_symbol ?? null)
            ?? ''
        );
    }

    /**
     * Get network fee if available.
     */
    private function getFee(): ?float
    {
        return $this->transaction->metadata['network_fee'] ?? null;
    }
}
