<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\StakingPlan;
use App\Models\UserStake;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\Notification;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\StakingStartedEmail;
use App\Services\UserActivityService;

class StakingController extends Controller
{
    public function index()
    {
        $plans = StakingPlan::with('coin')->where('is_active', true)->get();
        $userStakes = auth()->user()->userStakes()->with('stakingPlan.coin')->get();
        return view('user.staking.index', compact('plans', 'userStakes'));
    }

    public function show(StakingPlan $stakingPlan)
    {
        $user = auth()->user();
        $coin = $stakingPlan->coin;
        $userBalance = $user->coinBalances()->where('coin_id', $coin->id)->where('is_enabled', true)->first();
        $balance = $userBalance ? $userBalance->balance : 0;
        $userStake = $user->userStakes()->where('staking_plan_id', $stakingPlan->id)->where('status', 'active')->first();
        return view('user.staking.show', compact('stakingPlan', 'balance', 'userStake'));
    }

    public function stake(Request $request, StakingPlan $stakingPlan)
    {
        $user = auth()->user();
        $coin = $stakingPlan->coin;
        $userBalance = $user->coinBalances()->where('coin_id', $coin->id)->where('is_enabled', true)->first();
        $balance = $userBalance ? $userBalance->balance : 0;

        $data = $request->validate([
            'amount' => 'required|numeric|min:' . $stakingPlan->min_amount . '|max:' . $stakingPlan->max_amount,
        ]);

        if ($balance < $data['amount']) {
            return back()->withErrors(['amount' => 'Insufficient balance.']);
        }

        $activityData = [
            'amount' => $data['amount'],
            'currency' => $coin->symbol,
            'plan_name' => $stakingPlan->name,
            'roi' => $stakingPlan->return_of_interest,
            'interest_type' => $stakingPlan->interest_type,
            'lock_duration' => $stakingPlan->lock_duration,
        ];

        return UserActivityService::logInTransaction(
            'stake',
            $user,
            function () use ($user, $coin, $userBalance, $stakingPlan, $data) {
                DB::beginTransaction();
                try {
                    // Deduct balance
                    $userBalance->balance -= $data['amount'];
                    $userBalance->save();

                    // Create user stake
                    $start = now();
                    $end = now()->addDays($stakingPlan->lock_duration);
                    $userStake = UserStake::create([
                        'user_id' => $user->id,
                        'staking_plan_id' => $stakingPlan->id,
                        'amount' => $data['amount'],
                        'start_time' => $start,
                        'end_time' => $end,
                        'status' => 'active',
                        'profit_earned' => 0,
                    ]);

                    // Create transaction
                    $transaction = Transaction::create([
                        'user_id' => $user->id,
                        'from_coin_id' => $coin->id,
                        'type' => 'stake',
                        'amount_in' => $data['amount'],
                        'status' => 'completed',
                        'metadata' => [
                            'staking_plan_id' => $stakingPlan->id,
                            'plan_name' => $stakingPlan->name,
                            'roi' => $stakingPlan->return_of_interest,
                            'interest_type' => $stakingPlan->interest_type,
                            'lock_duration' => $stakingPlan->lock_duration,
                        ],
                        'processed_at' => now(),
                    ]);

                    // Create notification
                    Notification::create([
                        'user_id' => $user->id,
                        'type' => 'staking',
                        'title' => 'Staking Started',
                        'message' => 'You have staked ' . number_format($data['amount'], 8) . ' ' . $coin->display_symbol . ' in ' . $stakingPlan->name,
                        'is_read' => false,
                        'extra_data' => json_encode([
                            'staking_plan_id' => $stakingPlan->id,
                            'user_stake_id' => $userStake->id,
                            'amount' => $data['amount'],
                            'start_time' => $start,
                            'end_time' => $end,
                        ]),
                    ]);

                    // Send staking started email
                    Mail::to($user->email)->queue(new StakingStartedEmail($userStake));

                    DB::commit();
                    return $transaction;
                } catch (\Exception $e) {
                    DB::rollBack();
                    \Log::error('Staking failed: ' . $e->getMessage(), [
                        'user_id' => $user->id,
                        'coin_id' => $coin->id ?? null,
                        'staking_plan_id' => $stakingPlan->id,
                        'user_balance' => $userBalance,
                        'data' => $data,
                        'trace' => $e->getTraceAsString(),
                    ]);
                    throw $e;
                }
            },
            $activityData
        ) ? redirect()->route('staking.success', ['userStake' => $userStake->id]) : back()->withErrors(['error' => 'Failed to stake. Please try again.']);
    }

    public function success(UserStake $userStake)
    {
        return view('user.staking.success', compact('userStake'));
    }
} 