<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use App\Services\UserActivityService;
use App\Services\UserCoinService;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\User;
use App\Models\Transaction;
use App\Models\Notification;
use App\Mail\TransactionEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class SendController extends Controller
{
    private CryptoService $cryptoService;
    private UserCoinService $userCoinService;

    public function __construct(CryptoService $cryptoService, UserCoinService $userCoinService)
    {
        $this->cryptoService = $cryptoService;
        $this->userCoinService = $userCoinService;
    }

    /**
     * Show user's PayID assets page.
     */
    public function payidAssets()
    {
        $user = Auth::user();
        $hasBalances = $user->coinBalances()
            ->whereHas('coin', fn($query) => $query->where('is_active', true))
            ->where('is_enabled', true)
            ->exists();

        if (!$hasBalances) {
            return redirect()->route("dashboard")->with("error", "No crypto assets found");
        }

        return view("user.send.assets-payid", [
            "assets" => $this->mapCryptoAssets(),
        ]);
    }

    /**
     * Show details for a specific PayID asset.
     */
    public function payidDetails($symbol, $network = "native")
    {
        $user = Auth::user();

        if ($user->require_kyc && $user->kyc_status !== 'approved') {
            return redirect()->route('kyc')->with('error', 'Please verify your KYC to send crypto');
        }

        $asset = $this->getAssetDetails($symbol, $network);

        if (!$asset) {
            return redirect()->route("dashboard")->with("error", "Asset not found");
        }

        return view("user.send.payid-details", ["asset" => $asset]);
    }

    /**
     * Format coin asset for user view.
     */
    private function formatAsset(Coin $coin, UserCoinBalance $balance = null)
    {
        $prices = $this->cryptoService->getPrices();
        $price = $coin->getEffectivePrice();
        $balanceAmount = $balance ? $balance->balance : 0;

        $change = 0;
        $coinPriceData = collect($prices)->first(function ($priceData, $key) use ($coin) {
            return stripos($key, $coin->symbol) !== false ||
                   stripos($key, $coin->price_symbol ?? $coin->symbol) !== false;
        });

        if ($coinPriceData && isset($coinPriceData['usd_24h_change'])) {
            $change = $coinPriceData['usd_24h_change'];
        }

        return [
            "symbol" => $coin->symbol,
            "name" => $coin->display_name ?? ($coin->name ?? $coin->symbol),
            "balance" => $balanceAmount,
            "value" => $balanceAmount * $price,
            "price" => $price,
            "change" => $change < 0 ? number_format($change, 2) : "+" . number_format($change, 2),
            "icon_url" => $coin->image_url ?? null,
            "network" => $coin->network,
            "network_url" => $coin->image_url ?? null,
        ];
    }

    /**
     * Map user's active crypto assets.
     */
    private function mapCryptoAssets()
    {
        $user = Auth::user();
        $assets = [];

        $userBalances = $user->coinBalances()
            ->with('coin')
            ->whereHas('coin', fn($query) => $query->where('is_active', true))
            ->where('is_enabled', true)
            ->get();

        foreach ($userBalances as $balance) {
            if (!$balance->coin) continue;
            $assets[] = $this->formatAsset($balance->coin, $balance);
        }

        return $assets;
    }

    /**
     * Show external assets for the user.
     */
    public function externalAssets()
    {
        $user = Auth::user();

        $hasBalances = $user->coinBalances()
            ->whereHas('coin', fn($query) => $query->where('is_active', true))
            ->where('is_enabled', true)
            ->exists();

        if (!$hasBalances) {
            return redirect()->route("dashboard")->with("error", "No crypto assets found");
        }

        return view("user.send.assets-external", [
            "assets" => $this->mapCryptoAssets(),
        ]);
    }

    public function externalDetails($symbol, $network = "native")
    {
        $user = Auth::user();

        if ($user->require_kyc && $user->kyc_status !== 'approved') {
            return redirect()->route('kyc')->with('error', 'Please verify your KYC to send crypto');
        }

        $asset = $this->getAssetDetails($symbol, $network);

        if (!$asset) {
            return redirect()->route("dashboard")->with("error", "Asset not found");
        }

        return view("user.send.external-details", ["asset" => $asset]);
    }

    /**
     * Verify external address format.
     */
    public function verifyExternal(Request $request)
    {
        $request->validate([
            'address' => 'required|string',
            'symbol' => 'required|string',
            'network' => 'nullable|string'
        ]);

        $address = $request->address;
        $symbol = strtoupper($request->symbol);
        $network = strtoupper($request->network ?? 'native');

        $coin = $this->getCoinBySymbolNetwork($symbol, $network);
        if (!$coin) {
            return response()->json(['success' => false, 'message' => 'Cryptocurrency not found or inactive.']);
        }

        $isValid = $this->validateCryptoAddress($address, $coin);

        return response()->json([
            'success' => $isValid,
            'message' => $isValid ? 'Address is valid.' : 'Invalid address or incorrect network.'
        ]);
    }

    /**
     * Validate crypto address against coin type.
     */
    private function validateCryptoAddress($address, $coin)
    {
        if ($coin instanceof Coin) {
            return method_exists($coin, 'validateAddress') ? $coin->validateAddress($address) : false;
        }
        return false;
    }

    /**
     * Handle external crypto withdrawal (send).
     */
    public function processExternal(Request $request, $symbol, $network = 'native')
    {
        $validated = $request->validate([
            'address' => 'required|string',
            'amount' => 'required|numeric|min:0',
        ]);

        try {
            $sender = Auth::user();
            $address = $request->address;
            $amount = (float) $request->amount;

            $coin = $this->getCoinBySymbolNetwork($symbol, $network);
            if (!$coin) {
                return redirect()->route('send.external.details', compact('symbol', 'network'))
                    ->with(['error' => 'Cryptocurrency not found or inactive']);
            }

            $isValid = $this->validateCryptoAddress($address, $coin);
            if (!$isValid) {
                return redirect()->route('send.external.details', compact('symbol', 'network'))
                    ->with(['error' => 'Invalid address format or incorrect network.']);
            }

            $senderBalance = $this->userCoinService->getUserCoinBalance($sender, $coin);
            if (!$senderBalance || !$senderBalance->is_enabled) {
                return redirect()->route('send.external.details', compact('symbol', 'network'))
                    ->with(['error' => 'Cryptocurrency not enabled for sending']);
            }

            $networkFeeAmount = $senderBalance->getEffectiveFee();
            if ($senderBalance->balance < ($amount + $networkFeeAmount)) {
                return redirect()->route('send.external.details', compact('symbol', 'network'))
                    ->with(['error' => 'Insufficient balance (including network fee).']);
            }

            $transactionHash = $this->generateTransactionHash();
            $currentRate = $coin->getEffectivePrice();

            // Log and create transaction using service wrapper
            $transaction = UserActivityService::logInTransaction(
                'crypto_withdraw',
                $sender,
                function () use ($senderBalance, $amount, $networkFeeAmount, $sender, $address, $transactionHash, $currentRate, $coin) {
                    $senderBalance->balance -= ($amount + $networkFeeAmount);
                    $senderBalance->save();

                    $transaction = Transaction::create([
                        "user_id" => $sender->id,
                        "from_coin_id" => $coin->id,
                        "type" => Transaction::TYPE_WITHDRAWAL,
                        "amount_out" => $amount,
                        'rate' => $currentRate,
                        "network_fee" => $networkFeeAmount,
                        "status" => Transaction::STATUS_PENDING,
                        "metadata" => [
                            "external_address" => $address,
                            "network_fee" => $networkFeeAmount,
                            "transaction_hash" => $transactionHash,
                            "current_rate" => $currentRate,
                            "transfer_type" => "external"
                        ],
                        "processed_at" => now(),
                    ]);

                    Notification::create([
                        'user_id' => $sender->id,
                        'type' => 'external_transfer',
                        'title' => 'External Crypto Transfer',
                        'message' => sprintf(
                            'Sent %s %s to %s',
                            number_format($amount, 8),
                            $coin->display_symbol ?? $coin->symbol,
                            substr($address, 0, 6) . '...' . substr($address, -4)
                        ),
                        'is_read' => false,
                        'extra_data' => json_encode([
                            'transaction_id' => $transaction->id,
                            'amount' => $amount,
                            'coin_symbol' => $coin->symbol,
                            'transaction_hash' => $transactionHash
                        ])
                    ]);

                    // If TransactionEmail expects a Transaction instance, pass it.
                    try {
                        Mail::to($sender->email)->queue(new TransactionEmail($transaction));
                    } catch (\Throwable $mailEx) {
                        \Log::warning("Failed to queue transaction email: " . $mailEx->getMessage());
                    }

                    return $transaction;
                },
                [
                    'external_address' => $address,
                    'amount' => $amount,
                    'coin_id' => $coin->id,
                    'coin_symbol' => $coin->symbol,
                    'network_fee' => $networkFeeAmount,
                    'transaction_hash' => $transactionHash,
                    'transfer_type' => 'external',
                    'current_rate_usd' => $currentRate,
                    'value_usd' => $amount * $currentRate
                ]
            );

            return redirect()->route('send.external.success', [
                'symbol' => $symbol,
                'network' => $network,
                'transaction' => $transaction->id
            ])->with('success', 'Transaction submitted successfully.');
        } catch (\Exception $e) {
            \Log::error("External transfer failed: " . $e->getMessage());

            return redirect()->route('send.external.details', [
                'symbol' => $symbol,
                'network' => $network
            ])->with(['error' => 'Transaction failed: ' . $e->getMessage()]);
        }
    }

    /**
     * Generate a random 64-character hash.
     */
    private function generateTransactionHash()
    {
        return bin2hex(random_bytes(32));
    }

    /**
     * Success page for external transfers.
     */
    public function externalSuccess($symbol, $network, $transactionId)
    {
        $transaction = Transaction::where('id', $transactionId)
            ->where('user_id', Auth::id())
            ->firstOrFail();

        return view("user.send.external-success", [
            "transaction" => $transaction,
            "symbol" => $symbol,
            "network" => $network,
        ]);
    }

    /**
     * Failure page for external transfers.
     */
    public function externalFailed($symbol, $network)
    {
        return view("user.send.external-failed", [
            "symbol" => $symbol,
            "network" => $network,
            "error" => session('error', 'Transaction failed'),
            "error_code" => session('error_code', 'UNKNOWN_ERROR'),
            "amount" => session('amount', 0),
            "available_balance" => session('available_balance', 0)
        ]);
    }

    /**
     * Get coin by symbol and network.
     */
    private function getCoinBySymbolNetwork($symbol, $network = "native")
    {
        $query = Coin::where('symbol', strtoupper($symbol))
            ->where('is_active', true);

        if ($network && strtoupper($network) !== 'NATIVE') {
            $query->where('network', strtoupper($network));
        } else {
            $query->whereNull('network');
        }

        return $query->first();
    }

    /**
     * Get asset details for a user formatted for view.
     * Returns array|null (formatted asset) to match your original view usage.
     */
    private function getAssetDetails($symbol, $network = "native")
    {
        $user = Auth::user();

        $query = Coin::where('symbol', strtoupper($symbol))
            ->where('is_active', true);

        if ($network && strtoupper($network) !== 'NATIVE') {
            $query->where('network', strtoupper($network));
        } else {
            $query->whereNull('network');
        }

        $coin = $query->first();
        if (!$coin) return null;

        $balance = $user->coinBalances()
            ->where('coin_id', $coin->id)
            ->where('is_enabled', true)
            ->first();

        if (!$balance) return null;

        $asset = $this->formatAsset($coin, $balance);
        $asset['fee'] = $balance->getEffectiveFee();
        $asset['price'] = $coin->getEffectivePrice();

        return $asset;
    }
}
