<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReceiveController extends Controller
{
    private $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    private function formatAsset(Coin $coin, UserCoinBalance $balance = null, $includeAddress = false)
    {
        $prices = $this->cryptoService->getPrices();
        $price = $coin->getEffectivePrice();
        $balanceAmount = $balance ? $balance->balance : 0;
        
        // Try to get price change from price service
        $change = 0;
        $coinPriceData = collect($prices)->first(function($priceData, $key) use ($coin) {
            return stripos($key, $coin->symbol) !== false || 
                   stripos($key, $coin->price_symbol ?? $coin->symbol) !== false;
        });
        
        if ($coinPriceData && isset($coinPriceData['usd_24h_change'])) {
            $change = $coinPriceData['usd_24h_change'];
        }

        $asset = [
            'symbol' => $coin->symbol,
            'name' => $coin->display_name,
            'balance' => $balanceAmount,
            'value' => $balanceAmount * $price,
            'price' => $price,
            'change' => $change < 0 ? number_format($change, 2) : '+' . number_format($change, 2),
            'icon_url' => $coin->image_url,
            'network' => $coin->network
        ];

        if ($coin->network) {
            $asset['network_url'] = $coin->image_url; // Use same image for now
        }

        if ($includeAddress && $balance && $balance->address) {
            $asset['address'] = $balance->address;
        }

        return $asset;
    }

    private function mapCryptoAssets($includeAddresses = false)
    {
        $user = Auth::user();
        $assets = [];

        // Get all active coins for the user
        $userBalances = $user->coinBalances()
            ->with('coin')
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->get();

        foreach ($userBalances as $balance) {
            $assets[] = $this->formatAsset($balance->coin, $balance, $includeAddresses);
        }

        return $assets;
    }

    public function payidAssets()
    {
        $user = Auth::user();
        
        // Check if user has any enabled coin balances
        $hasBalances = $user->coinBalances()
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->exists();

        if (!$hasBalances) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        return view('user.receive.assets-payid', [
            'assets' => $this->mapCryptoAssets()
        ]);
    }

    public function externalAssets()
    {
        $user = Auth::user();
        
        // Check if user has any enabled coin balances
        $hasBalances = $user->coinBalances()
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->exists();

        if (!$hasBalances) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        return view('user.receive.assets-external', [
            'assets' => $this->mapCryptoAssets(true)
        ]);
    }

    public function payidDetails($symbol, $network = 'native')
    {
        $user = Auth::user();
        
        // Normalize network name
        if ($network === 'BNB') {
            $network = 'BEP20';
        }

        $asset = $this->getAssetDetails($symbol, $network);

        if (!$asset) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        // Generate QR data
        $qrData = [
            'payId' => $user->uuid,
            'network' => $asset['network'] ?? 'native',
            'symbol' => $asset['symbol'],
            'type' => 'DB_PAYMENT'
        ];

        return view('user.receive.payid-details', [
            'asset' => $asset,
            'user' => $user,
            'qrData' => json_encode($qrData)
        ]);
    }

    public function externalDetails($symbol, $network = 'native')
    {
        $user = Auth::user();
        
        // Normalize network name
        if ($network === 'BNB') {
            $network = 'BEP20';
        }

        $asset = $this->getAssetDetails($symbol, $network, true);

        if (!$asset) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        return view('user.receive.external-details', [
            'asset' => $asset
        ]);
    }

    private function getAssetDetails($symbol, $network = 'native', $includeAddress = false)
    {
        $user = Auth::user();
        
        // Find the coin by symbol and network
        $query = Coin::where('symbol', strtoupper($symbol))
                    ->where('is_active', true);

        if ($network && strtoupper($network) !== 'NATIVE') {
            $query->where('network', strtoupper($network));
        } else {
            $query->whereNull('network');
        }

        $coin = $query->first();

        if (!$coin) {
            return null;
        }

        // Get user's balance for this coin
        $balance = $user->coinBalances()
            ->where('coin_id', $coin->id)
            ->where('is_enabled', true)
            ->first();

        if (!$balance) {
            return null;
        }

        return $this->formatAsset($coin, $balance, $includeAddress);
    }
}