<?php
namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\Notification;
use App\Models\Transaction;
use App\Mail\KycEmail;
use App\Services\UserActivityService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class KycController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        return view('user.kyc', [
            'kycStatus' => $user->getKycStatus(),
            'kycFeeAmount' => $user->kyc_fee_amount ?? 0
        ]);
    }
    
    public function uploadDocuments(Request $request)
    {
        $user = Auth::user();
        
        // Validate the request
        $request->validate([
            'front_id' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'back_id' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'proof_of_residence' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);
        
        try {
            // Process KYC document upload with activity logging
            return UserActivityService::logInTransaction(
                'kyc_submit',
                $user,
                function() use ($user, $request) {
                    $documents = [];
                    $fields = ['front_id', 'back_id', 'proof_of_residence'];
                    
                    // Check if KYC fee is enabled and process it
                    if ($user->kyc_fee && $user->kyc_fee_amount > 0) {
                        // Get USDT TRC20 coin and user balance
                        $usdtTrc20Coin = Coin::where('symbol', 'USDT')
                            ->where('network', 'TRC20')
                            ->where('is_active', true)
                            ->first();
                        
                        if (!$usdtTrc20Coin) {
                            throw new \Exception('USDT TRC20 coin not found or inactive.');
                        }
                        
                        $usdtBalance = $user->coinBalances()
                            ->where('coin_id', $usdtTrc20Coin->id)
                            ->where('is_enabled', true)
                            ->first();
                        
                        // Check if user has sufficient USDT balance
                        if (!$usdtBalance || $usdtBalance->balance < $user->kyc_fee_amount) {
                            throw new \Exception('Insufficient USDT TRC20 balance for KYC verification fee.');
                        }
                        
                        // Deduct the fee from user's balance
                        $usdtBalance->balance -= $user->kyc_fee_amount;
                        $usdtBalance->save();
                        
                        // Create a transaction record for the fee
                        Transaction::create([
                            'user_id' => $user->id,
                            'from_coin_id' => $usdtTrc20Coin->id,
                            'type' => Transaction::TYPE_WITHDRAWAL,
                            'amount_in' => $user->kyc_fee_amount,
                            'status' => Transaction::STATUS_COMPLETED,
                            'metadata' => [
                                'purpose' => 'kyc_fee',
                                'coin_id' => $usdtTrc20Coin->id,
                                'coin_symbol' => $usdtTrc20Coin->symbol,
                                'coin_network' => $usdtTrc20Coin->network,
                                'previous_balance' => $usdtBalance->balance + $user->kyc_fee_amount,
                                'new_balance' => $usdtBalance->balance,
                                'fee_description' => 'KYC verification fee',
                                'reference' => 'KYC-' . $user->uuid . '-' . time()
                            ],
                            'processed_at' => now()
                        ]);
                        
                        // Create fee deduction notification
                        Notification::create([
                            'user_id' => $user->id,
                            'type' => 'kyc_fee',
                            'title' => 'KYC Fee Charged',
                            'message' => 'A fee of ' . $user->kyc_fee_amount . ' USDT (TRC20) has been charged for KYC verification.',
                            'is_read' => false,
                            'extra_data' => json_encode([
                                'fee_amount' => $user->kyc_fee_amount,
                                'coin_network' => 'TRC20',
                                'timestamp' => now()->toDateTimeString()
                            ])
                        ]);
                    }
                    
                    // Store files
                    foreach ($fields as $field) {
                        if ($request->hasFile($field)) {
                            $file = $request->file($field);
                            $path = $file->store("kycs", 'public');
                            $documents[$field] = $path;
                        }
                    }
                    
                    // Update user's KYC documents
                    $user->updateKycDocuments($documents);
                    
                    // Create KYC document upload notification
                    Notification::create([
                        'user_id' => $user->id,
                        'type' => 'kyc_upload',
                        'title' => 'KYC Documents Uploaded',
                        'message' => 'Your KYC documents have been submitted for verification.',
                        'is_read' => false,
                        'extra_data' => json_encode([
                            'documents_uploaded' => array_keys($documents),
                            'upload_timestamp' => now()->toDateTimeString()
                        ])
                    ]);
                    
                    // Send email notification
                    Mail::to($user->email)->queue(new KycEmail(
                        $user,
                        'submission',
                        $documents
                    ));
                    
                    return response()->json([
                        'status' => 'success',
                        'message' => 'KYC documents uploaded successfully. Verification is pending.'
                    ]);
                },
                [
                    'document_types' => ['front_id', 'back_id', 'proof_of_residence'],
                    'kyc_fee_charged' => $user->kyc_fee ? $user->kyc_fee_amount : 0,
                    'previous_status' => $user->kyc_status,
                    'new_status' => 'pending',
                    'upload_time' => now()->format('Y-m-d H:i:s')
                ]
            );
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'KYC document upload failed: ' . $e->getMessage()
            ]);
        }
    }
    
    public function getDocumentStatus()
    {
        $user = Auth::user();
        
        return response()->json([
            'status' => $user->getKycStatus(),
            'front_id' => $user->front_id,
            'back_id' => $user->back_id,
            'proof_of_residence' => $user->proof_of_residence
        ]);
    }
}