<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Services\CryptoService;
use App\Services\UserCoinService;
use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use Illuminate\Support\Facades\Auth;

class CryptoController extends Controller
{
    private $cryptoService;
    private $userCoinService;

    public function __construct(CryptoService $cryptoService, UserCoinService $userCoinService)
    {
        $this->cryptoService = $cryptoService;
        $this->userCoinService = $userCoinService;
    }

    public function details($symbol, $network = 'native')
    {
        $user = Auth::user();
        
        // Find the coin based on symbol and network
        $coin = Coin::where('symbol', strtoupper($symbol))
            ->where('is_active', true)
            ->when($network !== 'native', function($query) use ($network) {
                return $query->where('network', strtoupper($network));
            })
            ->when($network === 'native', function($query) {
                return $query->whereNull('network');
            })
            ->first();

        if (!$coin) {
            abort(404, 'Cryptocurrency not found');
        }

        // Get user's balance for this coin
        $userBalance = $this->userCoinService->getUserCoinBalance($user, $coin);
        
        // Get standard coin ID for prices
        $coinId = $coin->price_symbol ?? strtolower($coin->symbol);
        
        // Fetch transactions for this coin
        $transactions = Transaction::where('user_id', $user->id)
            ->ofCoin($coin->id)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get();
        
        // Default asset data
        $assetData = [
            'symbol' => $coin->symbol,
            'name' => $coin->name,
            'network' => $coin->network,
            'balance' => $userBalance ? $userBalance->balance : 0,
            'value' => 0,
            'price' => 0,
            'price_change_24h' => 0,
            'icon_url' => $coin->image_url,
            'address' => $userBalance ? $userBalance->address : null,
            'is_enabled' => $userBalance ? $userBalance->is_enabled : false,
        ];
        
        try {
            $prices = $this->cryptoService->getPrices();
            
            $price = $prices[$coinId]['usd'] ?? 0;
            $balance = $userBalance ? $userBalance->balance : 0;
            
            $assetData['price'] = $price;
            $assetData['value'] = $balance * $price;
            $assetData['price_change_24h'] = $prices[$coinId]['usd_24h_change'] ?? 0;
            
        } catch (\Exception $e) {
            logger()->error('Error fetching crypto data: ' . $e->getMessage());
        }

        return view('user.crypto.details', [
            'asset' => $assetData,
            'coin' => $coin,
            'transactions' => $transactions,
            'coinId' => $coinId,
        ]);
    }

    public function manage()
    {
        // Ensure user is authenticated
        if (!auth()->check()) {
            return redirect()->route('login');
        }

        $user = auth()->user();
        
        // Ensure user has coin balances for all active coins
        $this->userCoinService->ensureUserCoinBalances($user);

        // Get user's coin balances with coin information
        $userCoins = UserCoinBalance::where('user_id', $user->id)
            ->with('coin')
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->get();

        // Prepare crypto data for the view
        $cryptos = $userCoins->map(function($userCoin) {
            $coin = $userCoin->coin;
            return [
                'id' => $coin->id,
                'symbol' => $coin->symbol,
                'name' => $coin->name,
                'network' => $coin->network,
                'icon' => $coin->image_url,
                'status' => $userCoin->is_enabled,
                'address' => $userCoin->address,
                'balance' => $userCoin->balance,
            ];
        });

        return view('user.crypto.manage', [
            'cryptos' => $cryptos
        ]);
    }

    public function updateManage(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'coin_id' => 'required|integer|exists:coins,id',
            'status' => 'required|boolean'
        ]);

        try {
            $user = auth()->user();

            // Find the user's coin balance
            $userCoinBalance = UserCoinBalance::where('user_id', $user->id)
                ->where('coin_id', $validated['coin_id'])
                ->first();

            if (!$userCoinBalance) {
                return response()->json([
                    'success' => false,
                    'message' => 'Coin balance not found'
                ], 404);
            }

            // Update the status
            $userCoinBalance->is_enabled = $validated['status'];
            $userCoinBalance->save();

            $coin = $userCoinBalance->coin;

            return response()->json([
                'success' => true,
                'message' => 'Cryptocurrency status updated successfully',
                'coin_id' => $coin->id,
                'coin_symbol' => $coin->symbol,
                'status' => $validated['status']
            ]);

        } catch (\Exception $e) {
            // Log the error
            \Log::error('Crypto Manage Update Error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while updating cryptocurrency status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function address()
    {
        $user = Auth::user();
        
        // Ensure user has coin balances for all active coins
        $this->userCoinService->ensureUserCoinBalances($user);

        // Get user's active coin balances with coin information
        $activeCoins = UserCoinBalance::where('user_id', $user->id)
            ->where('is_enabled', true)
            ->with('coin')
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->get();

        // Format the data for the view
        $activeCryptos = $activeCoins->map(function($userCoin) {
            $coin = $userCoin->coin;
            return [
                'symbol' => $coin->symbol,
                'name' => $coin->name,
                'network' => $coin->network,
                'address' => $userCoin->address,
                'icon_url' => $coin->image_url,
                'balance' => $userCoin->balance,
            ];
        })->toArray();

        return view('user.crypto.address', [
            'activeCryptos' => $activeCryptos,
        ]);
    }
}