<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\CardTransaction;
use App\Models\Card;
use App\Models\Transaction;
use App\Models\Notification;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\CryptoService;
use App\Services\UserActivityService;
use App\Services\UserCoinService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CardController extends Controller
{
    protected $cryptoService;
    protected $userCoinService;

    public function __construct(CryptoService $cryptoService, UserCoinService $userCoinService)
    {
        $this->cryptoService = $cryptoService;
        $this->userCoinService = $userCoinService;
    }

    public function index()
    {
        $user = Auth::user();
        $card = $user->cards()->first();

        return view('user.card.index', ['card' => $card]);
    }

    public function request()
    {
        $user = Auth::user();
        
        // Check if the user already has a card
        if ($user->cards()->exists()) {
            return back()->with('error', 'You already have a card');
        }

        // Ensure user has coin balances for all active coins
        $this->userCoinService->ensureUserCoinBalances($user);

        // Create a new virtual card (no crypto_asset_id needed anymore)
        $card = $user->cards()->create([
            'card_holder' => $user->name,
            'card_number' => $this->generateCardNumber(),
            'expiry_month' => now()->addYears(3)->format('m'),
            'expiry_year' => now()->addYears(3)->format('y'),
            'cvv' => $this->generateCVV(),
            'billing_address' => '651 N Broad Street, Middletown, Delaware, US',
            'zip_code' => '19709',
            'status' => 'inactive',
            'card_type' => 'virtual',
            'last_four' => substr($this->generateCardNumber(), -4),
            'is_default' => true,
        ]);

        return back()->with('success', 'Card Request Successful');
    }

    public function freeze(Card $card)
    {
        $this->authorize('update', $card);
        $card->freeze();

        return back()->with('success', 'Card frozen successfully');
    }

    public function unfreeze(Card $card)
    {
        $this->authorize('update', $card);
        $card->unfreeze();

        return back()->with('success', 'Card unfrozen successfully');
    }

    public function delete(Card $card)
    {
        $this->authorize('delete', $card);

        try {
            DB::transaction(function () use ($card) {
                // Check if card has remaining balance
                if ($card->balance > 0) {
                    // Add the remaining balance to USDT ERC20 using new coin system
                    $usdtErc20 = Coin::where('symbol', 'USDT')
                        ->where('network', 'ERC20')
                        ->where('is_active', true)
                        ->first();

                    if ($usdtErc20) {
                        $userBalance = UserCoinBalance::firstOrCreate(
                            [
                                'user_id' => $card->user_id,
                                'coin_id' => $usdtErc20->id
                            ],
                            [
                                'balance' => 0,
                                'is_enabled' => true
                            ]
                        );

                        $userBalance->increment('balance', $card->balance);
                    }
                }

                // Delete the card
                $card->delete();
            });

            return back()->with('success', 'Card deleted successfully.');
        } catch (\Exception $e) {
            \Log::error('Card deletion failed: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete card. Please try again.');
        }
    }

    public function addMoney()
    {
        $user = Auth::user();
        $card = $user->cards()->first();
        
        // Get user's active coin balances instead of CryptoAsset
        $userCoins = $this->userCoinService->getUserActiveCoins($user);
        $prices = $this->cryptoService->getPrices();

        return view('user.card.add-money', [
            'card' => $card,
            'userCoins' => $userCoins,
            'prices' => $prices,
        ]);
    }

    public function fund(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'coin_id' => 'required|integer|exists:coins,id',
        ]);

        $user = Auth::user();
        $card = $user->cards()->first();
        $prices = $this->cryptoService->getPrices();
        
        // Get the coin and user balance
        $coin = Coin::findOrFail($request->coin_id);
        $userBalance = UserCoinBalance::where('user_id', $user->id)
            ->where('coin_id', $coin->id)
            ->first();

        if (!$userBalance) {
            return back()->with('error', 'Coin balance not found');
        }

        // Get coin price using the coin's price symbol
        $priceKey = $coin->price_symbol ?? strtolower($coin->symbol);
        $coinPrice = $prices[$priceKey]['usd'] ?? 0;
        
        if ($coinPrice === 0) {
            return back()->with('error', 'Unable to get current price for ' . $coin->symbol);
        }

        // Calculate required crypto amount
        $requiredCryptoAmount = $request->amount / $coinPrice;
        
        // Check if the user has enough balance
        if ($userBalance->balance < $requiredCryptoAmount) {
            return back()->with('error', 'Insufficient ' . $coin->symbol . ' balance');
        }

        try {
            // Deduct from crypto balance and add to card balance using UserActivityService
            UserActivityService::logInTransaction(
                'card_funding',
                $user,
                function () use ($userBalance, $card, $request, $requiredCryptoAmount, $coin, $coinPrice, $user) {
                    // Update crypto balance
                    $userBalance->decrement('balance', $requiredCryptoAmount);
                    
                    // Update card balance
                    $card->balance += $request->amount;
                    $card->save();
                    
                    // Create card transaction record
                    $cardTransaction = CardTransaction::create([
                        'card_id' => $card->id,
                        'amount' => $request->amount,
                        'type' => 'credit',
                        'description' => sprintf(
                            'Funded from %s (%s %s)', 
                            $coin->name,
                            number_format($requiredCryptoAmount, 8),
                            $coin->symbol
                        )
                    ]);
                    
                    // Create crypto transaction record using new system
                    $transaction = Transaction::create([
                        'user_id' => $user->id,
                        'from_coin_id' => $coin->id,  // Use from_coin_id for the new coin system
                        'type' => Transaction::TYPE_FUNDING,
                        'amount_in' => $requiredCryptoAmount,
                        'status' => Transaction::STATUS_COMPLETED,
                        'metadata' => [
                            'card_id' => $card->id,
                            'usd_value' => $request->amount,
                            'price' => $coinPrice,
                            'card_amount' => $request->amount,
                            'coin_symbol' => $coin->symbol,
                            'coin_network' => $coin->network,
                            'purpose' => 'card_funding'
                        ],
                        'processed_at' => now()
                    ]);

                    // Create notification for card funding
                    Notification::create([
                        'user_id' => $user->id,
                        'type' => 'card_funding',
                        'title' => 'Card Funded',
                        'message' => sprintf(
                            'You funded your card with $%s USD using %s %s',
                            number_format($request->amount, 2),
                            $coin->symbol,
                            $coin->network ? "({$coin->network})" : ''
                        ),
                        'is_read' => false,
                        'extra_data' => json_encode([
                            'transaction_id' => $transaction->id,
                            'card_transaction_id' => $cardTransaction->id,
                            'amount' => $request->amount,
                            'coin_id' => $coin->id,
                            'coin_symbol' => $coin->symbol,
                            'crypto_amount' => $requiredCryptoAmount
                        ])
                    ]);
                    
                    return $transaction;
                },
                [
                    'card_id' => $card->id,
                    'card_last_four' => $card->last_four,
                    'amount' => $request->amount,
                    'currency' => 'USD',
                    'source_currency' => $coin->symbol,
                    'source_amount' => $requiredCryptoAmount,
                    'exchange_rate' => $coinPrice
                ]
            );
            
            return back()->with('success', 'Card funded successfully');
        } catch (\Exception $e) {
            \Log::error('Card funding failed: ' . $e->getMessage());
            return back()->with('error', 'Failed to process transaction. Please try again.');
        }
    }

    private function generateCardNumber()
    {
        return '4187' . str_pad(mt_rand(0, 9999999999), 12, '0', STR_PAD_LEFT);
    }

    private function generateCVV()
    {
        // Generate a random 3-digit CVV
        return str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    }
}