<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\CryptoService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BuyController extends Controller
{
    private $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function index()
    {
        $user = Auth::user();
        
        // Check if user has any enabled coin balances
        $hasBalances = $user->coinBalances()
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->exists();

        if (!$hasBalances) {
            return redirect()->route('dashboard')->with('error', 'No crypto assets found');
        }

        $assets = $this->mapCryptoAssets();

        return view('user.buy.index', [
            'assets' => $assets
        ]);
    }

    private function formatAsset(Coin $coin, UserCoinBalance $balance = null)
    {
        $prices = $this->cryptoService->getPrices();
        $price = $coin->getEffectivePrice();
        $balanceAmount = $balance ? $balance->balance : 0;
        
        // Try to get price change from price service
        $change = 0;
        $coinPriceData = collect($prices)->first(function($priceData, $key) use ($coin) {
            return stripos($key, $coin->symbol) !== false || 
                   stripos($key, $coin->price_symbol ?? $coin->symbol) !== false;
        });
        
        if ($coinPriceData && isset($coinPriceData['usd_24h_change'])) {
            $change = $coinPriceData['usd_24h_change'];
        }

        return [
            'symbol' => $coin->symbol,
            'name' => $coin->display_name,
            'balance' => $balanceAmount,
            'value' => $balanceAmount * $price,
            'price' => $price,
            'change' => $change < 0 ? number_format($change, 2) : '+' . number_format($change, 2),
            'icon_url' => $coin->image_url,
            'network_url' => $coin->network ? $coin->image_url : null,
            'network' => $coin->network
        ];
    }

    private function mapCryptoAssets()
    {
        $user = Auth::user();
        $assets = [];

        // Get all active coins for the user
        $userBalances = $user->coinBalances()
            ->with('coin')
            ->whereHas('coin', function($query) {
                $query->where('is_active', true);
            })
            ->where('is_enabled', true)
            ->get();

        foreach ($userBalances as $balance) {
            $assets[] = $this->formatAsset($balance->coin, $balance);
        }

        return $assets;
    }

    public function details($symbol, $network = 'native')
    {
        $user = Auth::user();
        
        // Normalize network name
        if ($network === 'BNB') {
            $network = 'BEP20';
        }

        // Find the coin by symbol and network
        $query = Coin::where('symbol', strtoupper($symbol))
                    ->where('is_active', true);

        if ($network && $network !== 'native') {
            $query->where('network', strtoupper($network));
        } else {
            $query->whereNull('network');
        }

        $coin = $query->first();

        if (!$coin) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        // Get user's balance for this coin
        $balance = $user->coinBalances()
            ->where('coin_id', $coin->id)
            ->where('is_enabled', true)
            ->first();

        if (!$balance) {
            return redirect()->route('dashboard')->with('error', 'Asset not found');
        }

        $asset = $this->formatAsset($coin, $balance);

        return view('user.buy.details', [
            'asset' => $asset
        ]);
    }
}