<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Bot;
use App\Models\BotSubscription;
use App\Models\BotTrade;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Mail\BotSubscriptionEmail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class BotController extends Controller
{
   	public function index()
	{
    // Get active subscriptions for display
    $userSubscriptions = auth()->user()
        ->botSubscriptions()
        ->with(['bot', 'profits'])
        ->where('status', 'active')
        ->get();

    // Get ALL subscriptions for total calculations
    $allSubscriptions = auth()->user()
        ->botSubscriptions()
        ->get();

    // Get trades
    $trades = BotTrade::with('bot')
        ->whereHas('subscriptions', function($query) {
            $query->where('user_id', auth()->id());
        })
        ->latest()
        ->get();

    // Calculate win rate
    $totalTrades = $trades->count();
    $winRate = $totalTrades > 0 
        ? round(($trades->where('result', 'win')->count() / $totalTrades) * 100, 2)
        : 0;

    // Calculate totals from all subscriptions
    $totalInvestment = $allSubscriptions->sum('amount');
    $totalProfit = $allSubscriptions->sum('total_profit');

    return view('user.bots.index', compact(
        'userSubscriptions', 
        'trades', 
        'winRate',
        'totalInvestment',
        'totalProfit'
    ));
	}

    public function show(Bot $bot)
    {
        $userSubscription = auth()->user()
            ->botSubscriptions()
            ->where('bot_id', $bot->id)
            ->where('status', 'active')
            ->first();

        // Get user's USDT TRC20 balance using new coin system
        $user = auth()->user();
        $usdtTrc20Coin = Coin::where('symbol', 'USDT')
            ->where('network', 'TRC20')
            ->where('is_active', true)
            ->first();
        
        $usdtBalance = 0;
        if ($usdtTrc20Coin) {
            $balance = $user->coinBalances()
                ->where('coin_id', $usdtTrc20Coin->id)
                ->where('is_enabled', true)
                ->first();
            $usdtBalance = $balance ? $balance->balance : 0;
        }

        return view('user.bots.show', [
            'bot' => $bot,
            'userSubscription' => $userSubscription,
            'userBalance' => $usdtBalance
        ]);
    }

    public function subscribe(Request $request, Bot $bot)
    {
        Log::info('Starting bot subscription process', [
            'user_id' => auth()->id(),
            'request_data' => $request->all()
        ]);

        $data = $request->validate([
            'bot_type'     => 'required|string',
            'duration'     => 'required|string',
            'amount'       => 'required|numeric',
            'trading_pair' => 'required|string', 
        ]);

        // Find the bot
        $bot = Bot::where('bot_type', $data['bot_type'])->firstOrFail();
        Log::info('Bot found', ['bot_id' => $bot->id]);

        // Check min/max
        if ($data['amount'] < $bot->min_amount || $data['amount'] > $bot->max_amount) {
            Log::warning('Amount validation failed', [
                'provided' => $data['amount'],
                'min' => $bot->min_amount,
                'max' => $bot->max_amount
            ]);
            return response()->json([
                'success' => false,
                'message' => "Amount must be between {$bot->min_amount} and {$bot->max_amount}."
            ], 422);
        }

        // Check the user's USDT TRC20 balance using new coin system
        $user = auth()->user();
        $usdtTrc20Coin = Coin::where('symbol', 'USDT')
            ->where('network', 'TRC20')
            ->where('is_active', true)
            ->first();
        
        if (!$usdtTrc20Coin) {
            Log::error('USDT TRC20 coin not found or inactive');
            return response()->json([
                'success' => false,
                'message' => 'USDT TRC20 coin not available'
            ], 500);
        }
        
        $usdtBalance = $user->coinBalances()
            ->where('coin_id', $usdtTrc20Coin->id)
            ->where('is_enabled', true)
            ->first();
        
        $currentBalance = $usdtBalance ? $usdtBalance->balance : 0;

        if ($currentBalance < $data['amount']) {
            Log::warning('Insufficient balance', [
                'required' => $data['amount'],
                'available' => $currentBalance
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Insufficient USDT balance'
            ], 422);
        }

        DB::beginTransaction();
        try {
            // Deduct user's balance using new coin system
            if (!$usdtBalance) {
                // This shouldn't happen if user has balance, but safety check
                throw new \Exception('USDT balance record not found');
            }
            
            $previousBalance = $usdtBalance->balance;
            $usdtBalance->balance -= $data['amount'];
            $usdtBalance->save();
            
            Log::info('Balance deducted', [
                'amount' => $data['amount'],
                'previous_balance' => $previousBalance,
                'new_balance' => $usdtBalance->balance
            ]);

            // Create subscription record
            $subscription = BotSubscription::create([
                'bot_id'       => $bot->id,
                'user_id'      => $user->id,
                'amount'       => $data['amount'],
                'status'       => 'active',
                'subscribed_at'=> now(),
                'expires_at'   => $this->calculateExpiryDate($data['duration']),
            ]);
            Log::info('Subscription created', ['subscription_id' => $subscription->id]);
        	Mail::to($user->email)->queue(new BotSubscriptionEmail($subscription));

            // Create a BotTrade record
            $botTrade = BotTrade::create([
                'bot_id'       => $bot->id,
                'trading_pair' => $data['trading_pair'], 
                'action'       => 'subscribe',
                'amount'       => $data['amount'],
                'price'        => 0,
                'profit'       => 0,
                'metadata'     => [
                    'notes' => 'User subscribed to bot with dynamic pair.',
                ],
            ]);
            Log::info('Initial trade record created', ['trade_id' => $botTrade->id]);

            // Generate random action and fetch current price
            $randomAction = rand(0, 1) === 1 ? 'buy' : 'sell';
            $currentPrice = $this->fetchPriceFromApi($data['trading_pair']);

            // Update the trade with the fetched price
            $botTrade->update([
                'action' => $randomAction,
                'price'  => $currentPrice,
                'metadata' => array_merge($botTrade->metadata ?? [], [
                    'updated_for_trade' => now()->toDateTimeString(),
                ]),
            ]);
            Log::info('Trade record updated', [
                'trade_id' => $botTrade->id,
                'action' => $randomAction,
                'price' => $currentPrice
            ]);

            // Update bot's stats
            $bot->increment('total_subscribers');
            Log::info('Bot stats updated');

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Successfully subscribed to bot!'
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription process failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to subscribe. Please try again.'
            ], 500);
        }
    }

   protected function calculateExpiryDate(string $duration): ?\DateTime
	{
    return match ($duration) {
        // Minutes
        '5m'  => now()->addMinutes(5),
        '10m' => now()->addMinutes(10),
        '15m' => now()->addMinutes(15),
        '30m' => now()->addMinutes(30),

        // Hours
        '1h'  => now()->addHour(),
        '4h'  => now()->addHours(4),
        '12h' => now()->addHours(12),
        '24h' => now()->addDay(),

        // Days
        '3d'  => now()->addDays(3),
        '7d'  => now()->addDays(7),
        '30d' => now()->addDays(30),

       
        default => null,
    };
	}


   
    protected function fetchPriceFromApi(string $tradingPair): float
    {
        try {
            // Log the input
            Log::info('Fetching price from Huobi API', [
                'tradingPair' => $tradingPair
            ]);

            // Convert "BTC/USDT" to "btcusdt" for Huobi API
            $symbol = strtolower(str_replace('/', '', $tradingPair));
            Log::info('Formatted symbol for API', [
                'originalPair' => $tradingPair,
                'formattedSymbol' => $symbol
            ]);

            // Huobi's REST API endpoint
            $url = "https://api.huobi.pro/market/detail/merged?symbol={$symbol}";
            Log::info('Making API request', ['url' => $url]);

            // Make the request
            $response = Http::get($url);
            
            // Log the raw response
            Log::info('Raw API response', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);

            if (!$response->successful()) {
                Log::error('API request failed', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                throw new \Exception("Failed to fetch price for pair {$tradingPair} from Huobi API. Status: " . $response->status());
            }

            $data = $response->json();
            Log::info('Parsed JSON response', ['data' => $data]);

            if (isset($data['status']) && $data['status'] === 'ok' && isset($data['tick']['close'])) {
                $price = floatval($data['tick']['close']);
                Log::info('Successfully fetched price', [
                    'symbol' => $symbol,
                    'price' => $price
                ]);
                return $price;
            }

            Log::error('Invalid response structure', [
                'data' => $data,
                'symbol' => $symbol
            ]);
            throw new \Exception("Invalid price data received from Huobi API for {$tradingPair}");

        } catch (\Exception $e) {
            Log::error('Exception in fetchPriceFromApi', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'tradingPair' => $tradingPair
            ]);
            throw $e;
        }
    }

		public function updateTrades()
	{
    $trades = BotTrade::with('bot')
        ->whereHas('subscriptions', function($query) {
            $query->where('user_id', auth()->id());
        })
        ->latest()
        ->get()
        ->map(function($trade) {
            return [
                'id' => $trade->id,
                'bot_name' => $trade->bot->name,
                'amount' => $trade->amount,
                'status' => 'completed',
                'result' => $trade->result,
                'created_at' => $trade->created_at->diffForHumans()
            ];
        });

    $userSubscriptions = auth()->user()
        ->botSubscriptions()
        ->with(['bot'])
        ->where('status', 'active')
        ->get()
        ->map(function($subscription) {
            return [
                'id' => $subscription->id,
                'bot_name' => $subscription->bot->name,
                'amount' => $subscription->amount,
                'status' => 'active',
                'result' => 'pending'
            ];
        });

    return response()->json([
        'trades' => $trades,
        'subscriptions' => $userSubscriptions
    ]);
	}

    
    public function profits(BotSubscription $subscription)
    {
        $profits = $subscription->profits()
            ->with('trade')
            ->latest()
            ->paginate(20);

        return view('user.bots.profits', compact('subscription', 'profits'));
    }


	public function history()
	{
    $trades = BotTrade::with('bot')
        ->whereHas('subscriptions', function ($query) {
            $query->where('user_id', auth()->id());
        })
        ->latest()
        ->paginate(10);

    return view('user.bots.history', compact('trades'));
	}


    /**
     * Get bot's real-time performance stats
     */
    public function stats(Bot $bot)
    {
        $stats = [
            'total_profit' => $bot->total_profit,
            'total_subscribers' => $bot->total_subscribers,
            'last_trade' => $bot->last_trade_at?->diffForHumans(),
            'current_pair' => $bot->trading_pair,
            'recent_trades' => $bot->trades()
                ->latest()
                ->take(5)
                ->get()
                ->map(fn($trade) => [
                    'action' => $trade->action,
                    'amount' => $trade->amount,
                    'price' => $trade->price,
                    'profit' => $trade->profit,
                    'time' => $trade->created_at->diffForHumans()
                ])
        ];

        return response()->json($stats);
    }
}