<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\Notification;
use App\Mail\WelcomeEmail;
use App\Models\Setting;
use App\Services\UserActivityService;
use App\Services\UserCoinService;
use Illuminate\Support\Facades\Mail;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use Illuminate\Support\Str;

class RegisteredUserController extends Controller
{
    protected $userCoinService;

    public function __construct(UserCoinService $userCoinService)
    {
        $this->userCoinService = $userCoinService;
    }

    public function create(Request $request): View
    {
        $referral_code = $request->ref;
        $referrer = null;
        
        if ($referral_code) {
            $referrer = User::where('referral_code', $referral_code)->first();
        }
        
        return view('auth.register', compact('referral_code', 'referrer'));
    }

    public function verifyReferralCode(Request $request)
    {
        $referrer = User::where('referral_code', $request->referral_code)->first();
        
        return response()->json([
            'valid' => !is_null($referrer),
            'referrer' => $referrer ? $referrer->name : null
        ]);
    }

    public function store(Request $request): RedirectResponse
{
    $request->validate([
        'name' => ['required', 'string', 'max:255'],
        'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
        'password' => ['required', 'confirmed', Rules\Password::defaults()],
        'referral_code' => ['nullable', 'string', 'exists:users,referral_code'],
    ]);

    try {
        $user = UserActivityService::logInTransaction(
            'register',
            new User(['email' => $request->email, 'name' => $request->name]),
            function () use ($request) {
                $referrer = null;
                if ($request->referral_code) {
                    $referrer = User::where('referral_code', $request->referral_code)->lockForUpdate()->first();
                }

                // Check if email verification is disabled in settings
                $emailVerificationRequired = setting('email_verification', true); // default: true

                $user = User::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => Hash::make($request->password),
                    'referral_code' => 'REF' . strtoupper(Str::random(8)),
                    'referred_by' => $referrer ? $referrer->id : null,
                    'referral_count' => 0,
                    'email_verified_at' => $emailVerificationRequired ? null : now(),
                ]);

                if ($referrer) {
                    $referrer->increment('referral_count');

                    if (
                        setting('referral_enabled') &&
                        (
                            setting('referral_bonus_type') === 'instant' ||
                            (setting('referral_bonus_type') === 'after_kyc' && $referrer->kyc_status === 'approved')
                        )
                    ) {
                        $bonus_amount = (float) setting('referral_bonus_amount', 0);

                        $usdtTrc20Coin = Coin::where('symbol', 'USDT')
                            ->where('network', 'TRC20')
                            ->where('is_active', true)
                            ->first();

                        if ($usdtTrc20Coin) {
                            $referrerBalance = UserCoinBalance::firstOrCreate(
                                [
                                    'user_id' => $referrer->id,
                                    'coin_id' => $usdtTrc20Coin->id,
                                ],
                                [
                                    'balance' => 0,
                                    'is_enabled' => true,
                                ]
                            );

                            $referrerBalance->increment('balance', $bonus_amount);

                            Notification::createNotification([
                                'user_id' => $referrer->id,
                                'type' => 'referral_bonus',
                                'title' => 'Referral Bonus Received',
                                'message' => "You have received a referral bonus of $${bonus_amount} USDT for referring {$user->name}",
                                'extra_data' => json_encode([
                                    'amount' => $bonus_amount,
                                    'referred_user' => $user->name,
                                ]),
                            ]);
                        }
                    }
                }

                $this->userCoinService->createUserCoinBalances($user);

                Mail::to($user->email)->queue(new WelcomeEmail($user));

                return $user;
            },
            [
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'referred_by' => $request->referral_code,
            ]
        );

        event(new Registered($user));
        Auth::login($user);

        return redirect(route('dashboard', absolute: false));
    } catch (\Exception $e) {
        return back()
            ->withInput($request->except('password'))
            ->withErrors(['registration' => 'Registration failed: ' . $e->getMessage()]);
    }
}

}