<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Models\Notification;
use App\Mail\TransactionStatusEmail;
use App\Services\UserCoinService;
use Illuminate\Support\Facades\Mail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TransactionController extends Controller
{
    protected $userCoinService;

    public function __construct(UserCoinService $userCoinService)
    {
        $this->userCoinService = $userCoinService;
    }

    /**
     * Display all transactions with optional filters.
     */
    public function index(Request $request)
    {
        $query = Transaction::with(['user', 'fromCoin', 'toCoin'])->latest();

        // Filters
        if ($request->filled('type')) {
            $query->ofType($request->type);
        }
        if ($request->filled('status')) {
            $query->ofStatus($request->status);
        }
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transaction_hash', 'like', "%{$search}%")
                  ->orWhere('from_address', 'like', "%{$search}%")
                  ->orWhere('to_address', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($uq) use ($search) {
                      $uq->where('name', 'like', "%{$search}%")
                         ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $transactions = $query->paginate(10)->withQueryString();

        return view('admin.transactions.index', compact('transactions'));
    }

    /**
     * Create a new transaction (admin-side).
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id'        => 'required|exists:users,id',
            'type'           => 'required|in:' . implode(',', Transaction::getValidTypes()),
            'from_coin_id'   => 'required|exists:coins,id',
            'to_coin_id'     => 'nullable|required_if:type,swap|exists:coins,id',
            'transaction_hash' => 'nullable|string',
            'from_address'   => 'nullable|string',
            'to_address'     => 'nullable|string',
            'amount_in'      => 'required_unless:type,withdrawal|nullable|numeric|min:0',
            'amount_out'     => 'required_if:type,swap,withdrawal|nullable|numeric|min:0',
            'network_fee'    => 'nullable|numeric|min:0',
            'rate'           => 'nullable|numeric|min:0',
            'status'         => 'required|in:' . implode(',', Transaction::getValidStatuses()),
            'metadata'       => 'nullable|array',
        ]);

        try {
            DB::beginTransaction();

            $user = User::findOrFail($validated['user_id']);
            $fromCoin = Coin::findOrFail($validated['from_coin_id']);

            $fromBalance = $this->userCoinService->getUserCoinBalance($user, $fromCoin);
            if (!$fromBalance) {
                throw new \Exception('User does not have balance record for ' . $fromCoin->display_name);
            }

            // Ensure amount_out is set for withdrawals (fix for 0.00 issue)
            if (($validated['type'] ?? null) === Transaction::TYPE_WITHDRAWAL && empty($validated['amount_out'])) {
                $validated['amount_out'] = $validated['amount_in'] ?? 0;
            }

            $toBalance = null;
            if (!empty($validated['to_coin_id'])) {
                $toCoin = Coin::findOrFail($validated['to_coin_id']);
                $toBalance = $this->userCoinService->getUserCoinBalance($user, $toCoin);
                if (!$toBalance) {
                    throw new \Exception('User does not have balance record for ' . $toCoin->display_name);
                }
            }

            // Create the transaction
            $transaction = Transaction::create($validated);

            // Notify user for pending status
            if ($transaction->isPending()) {
                $this->createTransactionNotification($transaction, 'pending');
            }

            // If completed, immediately apply balance and send email
            if ($transaction->isCompleted()) {
                $this->updateCoinBalances($transaction);
                $this->createTransactionNotification($transaction, 'completed');
            }

            DB::commit();

            return back()->with('status', 'Transaction created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create transaction: ' . $e->getMessage());
        }
    }

    /**
     * Approve pending transactions (admin action).
     */
    public function approve(Transaction $transaction)
    {
        try {
            DB::beginTransaction();

            if (!$transaction->isPending()) {
                throw new \Exception('Only pending transactions can be approved.');
            }

            $transaction->status = Transaction::STATUS_COMPLETED;

            // Ensure withdrawal amount is valid on approval too
            if ($transaction->type === Transaction::TYPE_WITHDRAWAL && empty($transaction->amount_out)) {
                $transaction->amount_out = $transaction->amount_in ?? 0;
            }

            $transaction->save();

            $this->updateCoinBalances($transaction);
            $this->createTransactionNotification($transaction, 'completed');

            DB::commit();

            return back()->with('status', 'Transaction approved successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to approve transaction: ' . $e->getMessage());
        }
    }

    /**
     * Reject a pending transaction.
     */
    public function reject(Request $request, Transaction $transaction)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        try {
            if (!$transaction->isPending()) {
                throw new \Exception('Only pending transactions can be rejected.');
            }

            $transaction->status = Transaction::STATUS_FAILED;
            $transaction->metadata = array_merge($transaction->metadata ?? [], [
                'rejection_reason' => $request->reason,
                'rejected_at' => now(),
            ]);
            $transaction->save();

            // Send rejection email
            Mail::to($transaction->user->email)->queue(
                new TransactionStatusEmail($transaction, 'failed', $request->reason)
            );

            return back()->with('status', 'Transaction rejected successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to reject transaction: ' . $e->getMessage());
        }
    }

    /**
     * Delete a transaction (with reversal if necessary).
     */
    public function destroy(Transaction $transaction)
    {
        try {
            if ($transaction->isCompleted()) {
                $this->reverseCoinBalances($transaction);
            }

            $transaction->delete();

            return back()->with('status', 'Transaction deleted successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to delete transaction: ' . $e->getMessage());
        }
    }

    /**
     * Send notification + email for a transaction status.
     */
    protected function createTransactionNotification(Transaction $transaction, $status)
    {
        $user = $transaction->user;

        // Create notification entry
        $notificationData = [
            'user_id' => $user->id,
            'type' => 'send_crypto',
            'title' => ucfirst($status) . ' Transaction',
            'message' => $this->generateTransactionMessage($transaction, $status),
            'extra_data' => [
                'transaction_hash' => $transaction->transaction_hash,
                'status' => $status,
            ],
        ];

        Notification::createNotification($notificationData);

        // Send email
        Mail::to($user->email)->queue(
            new TransactionStatusEmail($transaction, $status)
        );
    }

    /**
     * Generate the human-readable message for a notification.
     */
    protected function generateTransactionMessage(Transaction $transaction, $status)
    {
        $coinSymbol = $transaction->fromCoin ? $transaction->fromCoin->display_symbol : 'unknown';

        switch ($transaction->type) {
            case Transaction::TYPE_DEPOSIT:
                return $status === 'pending'
                    ? "Incoming deposit of {$transaction->amount_in} {$coinSymbol} is pending."
                    : "Deposit of {$transaction->amount_in} {$coinSymbol} has been completed.";

            case Transaction::TYPE_WITHDRAWAL:
                $amount = $transaction->amount_out ?? $transaction->amount_in ?? 0;
                return $status === 'pending'
                    ? "Withdrawal of {$amount} {$coinSymbol} is pending."
                    : "Withdrawal of {$amount} {$coinSymbol} has been completed.";

            case Transaction::TYPE_SWAP:
                $fromSymbol = $transaction->fromCoin ? $transaction->fromCoin->display_symbol : 'unknown';
                $toSymbol = $transaction->toCoin ? $transaction->toCoin->display_symbol : 'unknown';
                return $status === 'pending'
                    ? "Swap of {$transaction->amount_in} {$fromSymbol} to {$transaction->amount_out} {$toSymbol} is pending."
                    : "Swap of {$transaction->amount_in} {$fromSymbol} to {$transaction->amount_out} {$toSymbol} has been completed.";

            case Transaction::TYPE_REFUND:
                return $status === 'pending'
                    ? "Refund of {$transaction->amount_in} {$coinSymbol} is pending."
                    : "Refund of {$transaction->amount_in} {$coinSymbol} has been completed.";

            case Transaction::TYPE_FUNDING:
                return $status === 'pending'
                    ? "Your funding of {$transaction->amount_in} {$coinSymbol} is pending."
                    : "Your funding of {$transaction->amount_in} {$coinSymbol} has been completed.";

            default:
                return 'Your transaction status has been updated.';
        }
    }

    /**
     * Update user coin balances after transaction completion.
     */
    protected function updateCoinBalances(Transaction $transaction)
    {
        $user = $transaction->user;

        switch ($transaction->type) {
            case Transaction::TYPE_DEPOSIT:
            case Transaction::TYPE_FUNDING:
            case Transaction::TYPE_REFUND:
                if ($transaction->fromCoin) {
                    $coinBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($coinBalance) {
                        $coinBalance->balance += $transaction->amount_in;
                        $coinBalance->save();
                    }
                }
                break;

            case Transaction::TYPE_WITHDRAWAL:
                if ($transaction->fromCoin) {
                    $coinBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($coinBalance) {
                        $coinBalance->balance -= $transaction->amount_in;
                        $coinBalance->save();
                    }
                }
                break;

            case Transaction::TYPE_SWAP:
                if ($transaction->fromCoin) {
                    $fromBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($fromBalance) {
                        $fromBalance->balance -= $transaction->amount_in;
                        $fromBalance->save();
                    }
                }
                if ($transaction->toCoin) {
                    $toBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->toCoin);
                    if ($toBalance) {
                        $toBalance->balance += $transaction->amount_out;
                        $toBalance->save();
                    }
                }
                break;
        }
    }

    /**
     * Reverse balances when deleting completed transactions.
     */
    protected function reverseCoinBalances(Transaction $transaction)
    {
        $user = $transaction->user;

        switch ($transaction->type) {
            case Transaction::TYPE_DEPOSIT:
            case Transaction::TYPE_FUNDING:
                if ($transaction->fromCoin) {
                    $coinBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($coinBalance) {
                        $coinBalance->balance -= $transaction->amount_in;
                        $coinBalance->save();
                    }
                }
                break;

            case Transaction::TYPE_WITHDRAWAL:
            case Transaction::TYPE_REFUND:
                if ($transaction->fromCoin) {
                    $coinBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($coinBalance) {
                        $coinBalance->balance += $transaction->amount_in;
                        $coinBalance->save();
                    }
                }
                break;

            case Transaction::TYPE_SWAP:
                if ($transaction->fromCoin) {
                    $fromBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->fromCoin);
                    if ($fromBalance) {
                        $fromBalance->balance += $transaction->amount_in;
                        $fromBalance->save();
                    }
                }
                if ($transaction->toCoin) {
                    $toBalance = $this->userCoinService->getUserCoinBalance($user, $transaction->toCoin);
                    if ($toBalance) {
                        $toBalance->balance -= $transaction->amount_out;
                        $toBalance->save();
                    }
                }
                break;
        }
    }
}
