<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\StakingPlan;
use App\Models\Coin;
use Illuminate\Http\Request;

class StakingPlanController extends Controller
{
    public function index()
    {
        $plans = StakingPlan::with('coin')->latest()->paginate(10);
        return view('admin.staking_plans.index', compact('plans'));
    }

    public function create()
    {
        $coins = Coin::where('is_active', true)->get();
        return view('admin.staking_plans.create', compact('coins'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|gt:min_amount',
            'return_of_interest' => 'required|numeric|min:0',
            'interest_type' => 'required|in:hourly,daily,weekly',
            'lock_duration' => 'required|integer|min:1',
            'coin_id' => 'required|exists:coins,id',
            'is_active' => 'boolean',
        ]);
        $validated['is_active'] = $request->has('is_active');
        StakingPlan::create($validated);
        return redirect()->route('admin.staking-plans.index')->with('status', 'Staking plan created successfully');
    }

    public function show(StakingPlan $stakingPlan)
    {
        $stakingPlan->load('coin', 'userStakes');
        return view('admin.staking_plans.show', compact('stakingPlan'));
    }

    public function edit(StakingPlan $stakingPlan)
    {
        $coins = Coin::where('is_active', true)->get();
        return view('admin.staking_plans.edit', compact('stakingPlan', 'coins'));
    }

    public function update(Request $request, StakingPlan $stakingPlan)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|gt:min_amount',
            'return_of_interest' => 'required|numeric|min:0',
            'interest_type' => 'required|in:hourly,daily,weekly',
            'lock_duration' => 'required|integer|min:1',
            'coin_id' => 'required|exists:coins,id',
            'is_active' => 'boolean',
        ]);
        $validated['is_active'] = $request->has('is_active');
        $stakingPlan->update($validated);
        return redirect()->route('admin.staking-plans.show', $stakingPlan)->with('status', 'Staking plan updated successfully');
    }

    public function destroy(StakingPlan $stakingPlan)
    {
        if ($stakingPlan->userStakes()->where('status', 'active')->exists()) {
            return back()->with('error', 'Cannot delete plan with active stakes');
        }
        $stakingPlan->delete();
        return redirect()->route('admin.staking-plans.index')->with('status', 'Staking plan deleted successfully');
    }
} 