<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Notification;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\UserCoinService;
use App\Mail\KycEmail;
use App\Mail\ReferralBonusEmail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class KycController extends Controller
{
    protected $userCoinService;

    public function __construct(UserCoinService $userCoinService)
    {
        $this->userCoinService = $userCoinService;
    }

    public function index(Request $request)
    {
        $query = User::where(function($query) {
            $query->whereNotNull('front_id')
                  ->whereNotNull('back_id')
                  ->whereNotNull('proof_of_residence');
        });
        
        if ($request->filled('status')) {
            $query->where('kyc_status', $request->status);
        }
        
        $users = $query->latest()
            ->paginate(10)
            ->withQueryString();
            
        return view('admin.kyc.index', compact('users'));
    }

    public function show(User $submission)
    {
        return view('admin.kyc.show', compact('submission'));
    }

    public function update(Request $request, $userId)
    {
        try {
            DB::beginTransaction();
            
            $user = User::findOrFail($userId);
            
            $validated = $request->validate([
                'status' => ['required', 'in:approved,rejected'],
                'rejection_reason' => ['required_if:status,rejected', 'nullable', 'string', 'max:500'],
            ]);

            $user->updateKycStatus($validated['status']);

            // Send email notification about status update
            Mail::to($user->email)->queue(new KycEmail(
                $user,
                'status_update',
                [],
                $validated['status'],
                $validated['rejection_reason'] ?? null
            ));

            // Handle referral bonus if KYC is approved
            if ($validated['status'] === 'approved' && 
                setting('referral_enabled') && 
                setting('referral_bonus_type') === 'after_kyc' && 
                $user->referred_by) {
                
                $referrer = User::find($user->referred_by);
                // Only give bonus if referrer hasn't received it yet (wasn't KYC verified at time of referral)
                if ($referrer && $referrer->kyc_status === 'approved') {
                    $bonus_amount = setting('referral_bonus_amount', 0);
                    
                    // Find USDT TRC20 coin
                    $usdtTrc20 = Coin::where('symbol', 'USDT')
                                    ->where('network', 'TRC20')
                                    ->where('is_active', true)
                                    ->first();
                    
                    if ($usdtTrc20) {
                        // Get or create user coin balance
                        $referrerBalance = $this->userCoinService->getUserCoinBalance($referrer, $usdtTrc20);
                        
                        if (!$referrerBalance) {
                            // Create the balance record if it doesn't exist
                            $this->userCoinService->ensureUserCoinBalances($referrer);
                            $referrerBalance = $this->userCoinService->getUserCoinBalance($referrer, $usdtTrc20);
                        }
                        
                        if ($referrerBalance) {
                            // Add the bonus amount
                            $referrerBalance->balance += $bonus_amount;
                            $referrerBalance->save();

                            // Create notification for referrer
                            Notification::createNotification([
                                'user_id' => $referrer->id,
                                'type' => 'referral_bonus',
                                'title' => 'Referral Bonus Received',
                                'message' => "You have received a referral bonus of {$bonus_amount} {$usdtTrc20->display_symbol} for referring {$user->name}",
                                'extra_data' => json_encode([
                                    'amount' => $bonus_amount,
                                    'coin_id' => $usdtTrc20->id,
                                    'coin_symbol' => $usdtTrc20->display_symbol,
                                    'referred_user' => $user->name
                                ])
                            ]);

                            // Send referral bonus email to referrer
                            Mail::to($referrer->email)->queue(new ReferralBonusEmail(
                                $referrer,
                                $user,
                                $bonus_amount
                            ));
                        }
                    }
                }
            }

            DB::commit();
            return back()->with('status', 'KYC submission has been ' . $validated['status']);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Failed to update KYC status: ' . $e->getMessage()]);
        }
    }
}