<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\CryptoService;
use Illuminate\Http\Request;

class CryptoController extends Controller
{
    protected CryptoService $cryptoService;

    public function __construct(CryptoService $cryptoService)
    {
        $this->cryptoService = $cryptoService;
    }

    public function show(User $user)
    {
        $activeCoins = $this->cryptoService->getActiveCoins();
        $userBalances = $user->coinBalances()->with('coin')->get()->keyBy('coin_id');
        
        return view('admin.users.crypto', [
            'user' => $user,
            'activeCoins' => $activeCoins,
            'userBalances' => $userBalances,
        ]);
    }

    public function update(Request $request, User $user)
    {
        // If we're just updating status
        if ($request->has('status_update')) {
            $request->validate([
                'crypto' => 'required|integer|exists:coins,id',
            ]);

            $coin = Coin::findOrFail($request->crypto);
            
            // Find or create user coin balance
            $balance = UserCoinBalance::firstOrCreate(
                ['user_id' => $user->id, 'coin_id' => $coin->id],
                ['balance' => 0, 'is_enabled' => true]
            );
            
            // Check for the dynamic status field name
            $statusFieldName = 'coin_' . $coin->id . '_status';
            $balance->update(['is_enabled' => $request->boolean($statusFieldName)]);

            return back()->with('status', 'Status updated successfully');
        }

        // Original balance update logic
        $request->validate([
            'crypto' => 'required|integer|exists:coins,id',
            'amount' => 'nullable|numeric|min:0',
            'action' => 'nullable|in:add,subtract',
        ]);

        $coin = Coin::findOrFail($request->crypto);
        
        // Find or create user coin balance
        $balance = UserCoinBalance::firstOrCreate(
            ['user_id' => $user->id, 'coin_id' => $coin->id],
            ['balance' => 0, 'is_enabled' => true]
        );
        
        if ($request->filled('amount') && $request->filled('action')) {
            $currentBalance = $balance->balance;
            $amount = $request->amount;

            if ($request->action === 'add') {
                $balance->update(['balance' => $currentBalance + $amount]);
            } else {
                if ($currentBalance >= $amount) {
                    $balance->update(['balance' => $currentBalance - $amount]);
                } else {
                    return back()->with('error', 'Insufficient balance for subtraction');
                }
            }
        }

        return back()->with('status', 'Crypto asset updated successfully');
    }
}