<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\UserCoinService;

class UpdateUserCoinAddresses extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'coins:update-addresses 
                            {--dry-run : Show what would be updated without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update user coin balance addresses that are null with current wallet settings';

    protected $userCoinService;

    /**
     * Create a new command instance.
     */
    public function __construct(UserCoinService $userCoinService)
    {
        parent::__construct();
        $this->userCoinService = $userCoinService;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Updating user coin addresses from wallet settings...');

        if ($this->option('dry-run')) {
            $this->info('DRY RUN MODE - No changes will be made');
            
            // Show what would be updated
            $nullAddresses = \App\Models\UserCoinBalance::whereNull('address')
                ->with(['coin', 'user'])
                ->get();
                
            if ($nullAddresses->isEmpty()) {
                $this->info('No null addresses found to update.');
                return 0;
            }
            
            $this->info("Found {$nullAddresses->count()} null addresses that would be updated:");
            
            foreach ($nullAddresses as $balance) {
                $settingKey = $this->userCoinService->generateWalletSettingKey($balance->coin);
                $walletAddress = \App\Models\Setting::where('key', $settingKey)->value('value');
                
                if ($walletAddress) {
                    $this->line("  - User: {$balance->user->email}, Coin: {$balance->coin->display_name} -> {$walletAddress}");
                } else {
                    $this->warn("  - User: {$balance->user->email}, Coin: {$balance->coin->display_name} -> No wallet address configured");
                }
            }
            
            return 0;
        }

        // Perform the actual update
        $updatedCount = $this->userCoinService->updateNullAddressesFromWalletSettings();

        if ($updatedCount > 0) {
            $this->info("Successfully updated {$updatedCount} null addresses with wallet settings.");
        } else {
            $this->info('No null addresses found to update.');
        }

        return 0;
    }
}
