<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Coin;
use App\Models\UserCoinBalance;
use App\Services\UserCoinService;
use Illuminate\Support\Facades\DB;

class ResetUserCoins extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'users:reset-coins 
                            {--dry-run : Show what would be done without making changes}
                            {--force : Force the operation without confirmation}';

    /**
     * The console command description.
     */
    protected $description = 'Delete all existing coin data for non-admin users and recreate using active coins';

    /**
     * Execute the console command.
     */
    public function handle(UserCoinService $userCoinService)
    {
        $isDryRun = $this->option('dry-run');
        $force = $this->option('force');

        $this->info('🚀 User Coin Reset Tool');
        $this->info('This will delete ALL coin/crypto fees, swap fees, balances, and addresses for non-admin users');
        $this->info('Then recreate them using the current active coins in the database');
        $this->newLine();

        // Get statistics
        $nonAdminUsers = User::where('is_admin', false)->get();
        $activeCoins = Coin::where('is_active', true)->get();
        $existingBalances = UserCoinBalance::whereIn('user_id', $nonAdminUsers->pluck('id'))->count();

        $this->table(['Metric', 'Count'], [
            ['Non-admin users', $nonAdminUsers->count()],
            ['Active coins', $activeCoins->count()],
            ['Existing coin balances', $existingBalances],
            ['New balances to create', $nonAdminUsers->count() * $activeCoins->count()],
        ]);

        $this->newLine();
        $this->info('Active coins that will be created for each user:');
        foreach ($activeCoins as $coin) {
            $network = $coin->network ? " ({$coin->network})" : '';
            $this->line("  • {$coin->symbol}{$network} - {$coin->name}");
        }

        if ($isDryRun) {
            $this->newLine();
            $this->warn('🔍 DRY RUN MODE - No changes will be made');
            return 0;
        }

        $this->newLine();
        if (!$force && !$this->confirm('⚠️  This will DELETE all existing coin data for non-admin users. Continue?')) {
            $this->info('Operation cancelled.');
            return 0;
        }

        $this->newLine();
        $this->info('🗑️  Deleting existing coin balances for non-admin users...');

        try {
            DB::transaction(function() use ($nonAdminUsers, $userCoinService) {
                // Delete existing UserCoinBalance records for non-admin users
                $deletedCount = UserCoinBalance::whereIn('user_id', $nonAdminUsers->pluck('id'))->delete();
                $this->info("   ✅ Deleted {$deletedCount} existing coin balance records");

                // Create new coin balances for all non-admin users
                $this->info('🔄 Creating new coin balances...');
                $progressBar = $this->output->createProgressBar($nonAdminUsers->count());
                $progressBar->start();

                foreach ($nonAdminUsers as $user) {
                    $userCoinService->createUserCoinBalances($user);
                    $progressBar->advance();
                }

                $progressBar->finish();
                $this->newLine();
            });

            $this->newLine();
            $this->info('✅ Successfully reset all user coin data!');
            
            // Show final statistics
            $newBalanceCount = UserCoinBalance::whereIn('user_id', $nonAdminUsers->pluck('id'))->count();
            $this->info("📊 Created {$newBalanceCount} new coin balance records");
            
            $this->newLine();
            $this->info('🎉 All non-admin users now have coin balances for all active cryptocurrencies');
            $this->info('   with addresses from wallet settings!');

        } catch (\Exception $e) {
            $this->error('❌ Error occurred: ' . $e->getMessage());
            return 1;
        }

        return 0;
    }
} 