<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\UserStake;
use App\Models\UserCoinBalance;
use App\Models\Transaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;

class FundStakingInterest extends Command
{
    protected $signature = 'staking:fund-interest';
    protected $description = 'Fund user accounts with staking interest and release principal when due';

    public function handle()
    {
        $now = now();
        $stakes = UserStake::with('stakingPlan.coin', 'user')
            ->where('status', 'active')
            ->get();
        $count = 0;
        foreach ($stakes as $stake) {
            $plan = $stake->stakingPlan;
            $coin = $plan->coin;
            $user = $stake->user;
            $userBalance = $user->coinBalances()->where('coin_id', $coin->id)->first();
            if (!$userBalance) continue;

            $start = Carbon::parse($stake->start_time);
            $end = Carbon::parse($stake->end_time);
            $now = now();

            // If next_payout is null, initialize it based on interest_type
            if (!$stake->next_payout) {
                if ($plan->interest_type === 'hourly') {
                    $stake->next_payout = $start->copy()->addHour();
                } elseif ($plan->interest_type === 'weekly') {
                    $stake->next_payout = $start->copy()->addWeek();
                } else { // default daily
                    $stake->next_payout = $start->copy()->addDay();
                }
                $stake->save();
            }

            // Pay periodic interest if due
            $intervalDue = false;
            $intervalInterest = 0;
            $interval = 0;
            if ($stake->next_payout && $now->greaterThanOrEqualTo($stake->next_payout) && $now->lessThan($end)) {
                $roi = $plan->return_of_interest;
                $duration = $end->diffInDays($start);
                if ($plan->interest_type === 'hourly') {
                    $interval = $end->diffInHours($start);
                    $intervalInterest = $stake->amount * ($roi / 100) / max($interval,1);
                    $stake->next_payout = $stake->next_payout->copy()->addHour();
                } elseif ($plan->interest_type === 'weekly') {
                    $interval = $end->diffInWeeks($start);
                    $intervalInterest = $stake->amount * ($roi / 100) / max($interval,1);
                    $stake->next_payout = $stake->next_payout->copy()->addWeek();
                } else { // daily
                    $interval = $duration;
                    $intervalInterest = $stake->amount * ($roi / 100) / max($interval,1);
                    $stake->next_payout = $stake->next_payout->copy()->addDay();
                }
                $intervalDue = true;
            }

            if ($intervalDue) {
                DB::beginTransaction();
                try {
                    $userBalance->balance += $intervalInterest;
                    $userBalance->save();
                    $stake->profit_earned += $intervalInterest;
                    $stake->save();
                    Transaction::create([
                        'user_id' => $user->id,
                        'from_coin_id' => $coin->id,
                        'type' => 'stake',
                        'amount_in' => $intervalInterest,
                        'status' => 'completed',
                        'metadata' => [
                            'staking_plan_id' => $plan->id,
                            'user_stake_id' => $stake->id,
                            'interest' => $intervalInterest,
                        ],
                        'processed_at' => now(),
                    ]);
                    DB::commit();
                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error('Staking interval interest failed: ' . $e->getMessage(), ['stake_id' => $stake->id]);
                }
            }

            // Release principal + any remaining interest at end
            if ($now->greaterThanOrEqualTo($end) && $stake->status === 'active') {
                $roi = $plan->return_of_interest;
                $total_interest = $stake->amount * ($roi / 100);
                $remaining_interest = $total_interest - $stake->profit_earned;
                $total = $stake->amount + $remaining_interest;
                DB::beginTransaction();
                try {
                    $userBalance->balance += $stake->amount + $remaining_interest;
                    $userBalance->save();
                    $stake->status = 'completed';
                    $stake->profit_earned = $total_interest;
                    $stake->next_payout = null;
                    $stake->save();
                    Transaction::create([
                        'user_id' => $user->id,
                        'from_coin_id' => $coin->id,
                        'type' => 'stake_release',
                        'amount_in' => $stake->amount + $remaining_interest,
                        'status' => 'completed',
                        'metadata' => [
                            'staking_plan_id' => $plan->id,
                            'user_stake_id' => $stake->id,
                            'interest' => $remaining_interest,
                            'principal' => $stake->amount,
                        ],
                        'processed_at' => now(),
                    ]);
                    DB::commit();
                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error('Staking release failed: ' . $e->getMessage(), ['stake_id' => $stake->id]);
                }
            }
        }
        $this->info("Processed $count completed stakes.");
    }
} 