<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ExchangeSupportChecker;
use App\Models\Coin;

class CheckCoinSupport extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'coins:check-support 
                           {coin? : Specific coin symbol to check}
                           {--all : Check all coins}
                           {--update : Update database with results}
                           {--timeout=30 : Timeout per exchange in seconds}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check cryptocurrency exchange support for coins';

    /**
     * Execute the console command.
     */
    public function handle(ExchangeSupportChecker $checker)
    {
        $coinSymbol = $this->argument('coin');
        $checkAll = $this->option('all');
        $updateDatabase = $this->option('update');

        if (!$coinSymbol && !$checkAll) {
            $this->error('Please specify a coin symbol or use --all to check all coins');
            return 1;
        }

        if ($coinSymbol) {
            return $this->checkSingleCoin($checker, $coinSymbol, $updateDatabase);
        }

        if ($checkAll) {
            return $this->checkAllCoins($checker, $updateDatabase);
        }

        return 0;
    }

    /**
     * Check support for a single coin
     */
    private function checkSingleCoin(ExchangeSupportChecker $checker, string $symbol, bool $update): int
    {
        $coin = Coin::where('symbol', strtoupper($symbol))->first();
        
        if (!$coin) {
            $this->error("Coin '{$symbol}' not found in database");
            return 1;
        }

        $this->info("Checking support for {$coin->name} ({$coin->symbol})...");
        $this->newLine();

        try {
            if ($update) {
                $result = $checker->updateCoinSupportStatus($coin);
                $this->displaySingleResult($result);
            } else {
                $results = $checker->checkCoinSupport($coin);
                $this->displayBasicResults($coin, $results);
            }
        } catch (\Exception $e) {
            $this->error("Error checking support: " . $e->getMessage());
            return 1;
        }

        return 0;
    }

    /**
     * Check support for all coins
     */
    private function checkAllCoins(ExchangeSupportChecker $checker, bool $update): int
    {
        $coins = Coin::active()->get();
        
        if ($coins->isEmpty()) {
            $this->warn('No active coins found in database');
            return 0;
        }

        $this->info("Checking support for {$coins->count()} active coins...");
        $this->newLine();

        $progressBar = $this->output->createProgressBar($coins->count());
        $progressBar->start();

        $results = [];
        $errors = 0;

        foreach ($coins as $coin) {
            try {
                if ($update) {
                    $result = $checker->updateCoinSupportStatus($coin);
                    $results[] = $result;
                } else {
                    $supportResults = $checker->checkCoinSupport($coin);
                    $results[] = [
                        'coin' => $coin,
                        'support_results' => $supportResults
                    ];
                }
            } catch (\Exception $e) {
                $errors++;
                $this->error("Error checking {$coin->symbol}: " . $e->getMessage());
            }

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine(2);

        // Display summary
        $this->displaySummary($results, $errors);

        return $errors > 0 ? 1 : 0;
    }

    /**
     * Display results for a single coin
     */
    private function displaySingleResult(array $result): void
    {
        $coin = $result['coin'];
        $status = $result['status'];
        $supportedCount = $result['supported_count'];
        $totalCount = $result['total_count'];

        $this->info("Results for {$coin->name} ({$coin->symbol}):");
        $this->info("Status: " . ucfirst($status));
        $this->info("Exchange Support: {$supportedCount}/{$totalCount}");
        $this->newLine();

        // Show exchange details
        $this->table(
            ['Exchange', 'Supported', 'Last Checked'],
            collect($result['support_results'])->map(function ($data, $exchange) {
                return [
                    ucfirst($exchange),
                    $data['supported'] ? '✅ Yes' : '❌ No',
                    $data['last_checked'] ?? 'Never'
                ];
            })->toArray()
        );

        // Show recommendations
        if (!empty($result['recommendations'])) {
            $this->newLine();
            $this->info('Recommendations:');
            foreach ($result['recommendations'] as $recommendation) {
                $icon = match($recommendation['type']) {
                    'critical' => '🔴',
                    'warning' => '🟡',
                    'info' => '🔵',
                    'success' => '🟢',
                    default => '📝'
                };
                $this->line($icon . ' ' . $recommendation['message']);
            }
        }
    }

    /**
     * Display basic results without database update
     */
    private function displayBasicResults(Coin $coin, array $results): void
    {
        $supportedCount = collect($results)->where('supported', true)->count();
        $totalCount = count($results);

        $this->info("Exchange Support: {$supportedCount}/{$totalCount}");
        $this->newLine();

        $this->table(
            ['Exchange', 'Supported', 'Last Checked'],
            collect($results)->map(function ($data, $exchange) {
                return [
                    ucfirst($exchange),
                    $data['supported'] ? '✅ Yes' : '❌ No',
                    $data['last_checked'] ?? 'Never'
                ];
            })->toArray()
        );
    }

    /**
     * Display summary for all coins
     */
    private function displaySummary(array $results, int $errors): void
    {
        $this->info('Summary:');
        $this->info("Total coins checked: " . count($results));
        $this->info("Errors encountered: {$errors}");
        $this->newLine();

        if (empty($results)) {
            return;
        }

        // Count by status
        $statusCounts = collect($results)
            ->groupBy('status')
            ->map->count()
            ->toArray();

        $this->table(
            ['Status', 'Count'],
            collect($statusCounts)->map(function ($count, $status) {
                return [ucfirst($status), $count];
            })->toArray()
        );

        // Show top supported and unsupported coins
        $supported = collect($results)->where('status', 'excellent')->take(5);
        $unsupported = collect($results)->where('status', 'unsupported')->take(5);

        if ($supported->isNotEmpty()) {
            $this->newLine();
            $this->info('Top Supported Coins:');
            foreach ($supported as $result) {
                $coin = $result['coin'];
                $this->line("✅ {$coin->name} ({$coin->symbol})");
            }
        }

        if ($unsupported->isNotEmpty()) {
            $this->newLine();
            $this->warn('Unsupported Coins:');
            foreach ($unsupported as $result) {
                $coin = $result['coin'];
                $this->line("❌ {$coin->name} ({$coin->symbol})");
            }
        }
    }
}
